%dw 2.0
/**
 * Returns a list of key value pair object describing the object entries
 *
 * .Example
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import dw::core::Objects
 * ---
 * Objects::entrySet({a: true, b: 1})
 * ----
 *
 * .Output
 * [source,json, linenums]
 * ----
 * [{key: "a", value: true}, {key: "b", value: 1}]
 * ----
 *
 */
fun entrySet<T <: Object>(obj: T) =
  obj pluck (value, key) -> {
    key: key,
    value: value,
    attributes: key.@
  }

/**
* Returns the list of key names of an object
*
* .Example
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::core::Objects
* ---
* Objects::nameSet({a: true, b: 1})
* ----
*
* .Output
* [source,json, linenums]
* ----
* ["a","b"]
* ----
*/
fun nameSet(obj: Object): Array<String> = obj pluck ($$ as String)

/**
* Returns the list of key names of an object
*
* .Example
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::core::Objects
* ---
* Objects::nameSet({a: true, b: 1})
* ----
*
* .Output
* [source,json, linenums]
* ----
* ["a","b"]
* ----
*/
fun keySet<T <: Object>(obj: T): ? = obj pluck $$

/**
* Returns the list of key values of an object
*
* .Example
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::core::Objects
* ---
* Objects::nameSet({a: true, b: 1})
* ----
*
* .Output
* [source,json, linenums]
* ----
* [true,1]
* ----
*/
fun valueSet <K,V>(obj: {K?: V}): Array<V> = obj pluck $

/**
 * Overrides the source with the target object so that the result with contain all the properties from the target
 * plus the properties on the source that are not declared on the target
 *
 * .Example
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import mergeWith from dw::core::Objects
 * ---
 * {a: true, b: 1} mergeWith {a: false, c: "Test"}
 * ----
 *
 * .Output
 * [source,json, linenums]
 * ----
 * {a: false, b: 1 , c: "Test"}
 * ----
 *
 */
fun mergeWith<T <: Object,V <: Object>(source: T, target: V): ? =
  (source -- keySet(target)) ++ target

/**
* Helper method to make `mergetWith` null friendly
*/
fun mergeWith<T <: Object>(a: Null, b: T): T = b

/**
* Helper method to make `mergetWith` null friendly
*/
fun mergeWith<T <: Object>(a: T, b: Null): T = a

/**
* Divides the object into sub objects with the specified amount of properties
*
* .Transform
* [source,DataWeave, linenums]
* ----
* output application/json
* ---
*  {a: 123,b: true, a:123,b:false} divideBy 2
* ----
*
* .Output
* [source,json, linenums]
* ----
* [
*    {
*      "a": 123,
*      "b": true
*    },
*    {
*      "a": 123,
*      "b": false
*    }
*  ]
* ----
*
*/
fun divideBy(items: Object, amount: Number): Array<{}> = do {
    fun internalDivideBy<T>(items: Object, amount: Number, carry:{} ): Array<{}> =
        items match {
          case {k:v ~ xs} ->
            if(sizeOf(carry) == amount - 1)
                [carry ++ {(k):v} ~ internalDivideBy(xs, amount, {})]
            else
               internalDivideBy(xs, amount, carry ++ {(k):v} )
          else ->
            if(isEmpty(carry))
             []
            else
             [carry]
        }
    ---
    internalDivideBy(items, amount, {})
}