%dw 2.0
import * from bat::Types
import DEBUG from bat::core::Console


var listBuffer = bat::Mutable::ListBuffer()

fun getSeqId() = using(a = listBuffer.push(1)) 'elem-$(listBuffer.length())'


var bullet = span @(class: 'bullet'): ''
var treeArrow = b @(class: 'tree-arrow'): '▶︎'

fun expander(target, inline = false) =
  // span @(class: 'tooltipped tooltipped-e', 'aria-tooltip': 'Expand details'):
    span @(class: "hidden-text-expander $(iif(inline, 'inline'))", onclick: 'ToggleClass("$target")'):
      button @('type':"button", class:"ellipsis-expander"): '…'

fun sanitizeName(name: String): String =
  name

fun itemWithContent(obj: {
  label: String,
  class: String,
  content: Any
}) = do {
  var contentId = getSeqId()
  ---
  div @(
    class: 'test $(obj.class)'
  ): {
    div @(class: 'test-name', onclick: 'selectContent("$contentId")'): {
      (bullet),
      span: obj.label
    },
    div @(class: 'content $contentId'): obj.content
  }
}




fun processHttp(elem: BATHttpStep, classname: String = '') = do {
  var time = (elem.time default 0.0) * 0.001
  var assertions = elem.assertions map print($, classname ++ '.' ++ sanitizeName(elem.name))
  var result = elem.result
  var hasReqPayload = result.request.payload? and (result.request.payload is String or result.request.payload is Binary)
  var hasResPayload = result.response.payload? and result.response.payload != null
  var rawId = getSeqId()
  var responseId = getSeqId()
  var requestId = getSeqId()
  ---
  itemWithContent({
    label: elem.name,
    class: passingClass(elem),
    content: {
      h2: elem.name,
      h3: 'Request',
      div @(class: 'raw request'): {
        div: {
          span @(class:"method"): result.request.method ++ ' ',
          a @(target: '_blank', href: result.request.url): result.request.path,
          span: ' ' ++ result.request.httpVersion,
        },
        (result.request.headers default {} pluck (div: {
          span @(class:"var"): $$ ++ ':',
          span: $
        })),
        (div: expander(requestId)) if hasReqPayload,
        (pre @(class:"req-res prettyprint $requestId"): (result.request.payload as String)) if hasReqPayload
      },
      h3: 'Response',
      div @(class: 'raw response'): {
        div: {
          span: result.request.httpVersion,
          span @(class:"var"): result.response.status,
          span: result.response.statusText,
        },
        (result.response.headers default {} pluck (div: {
          span @(class:"var"): $$ ++ ':',
          span: $
        })),
        (div: expander(responseId)) if hasResPayload,
        (pre @(class:"req-res prettyprint $responseId"): (result.response.payload as String)) if hasResPayload
      },
      h3: 'Assertions',
      (div @(class: 'assertions'): {( assertions )}) if sizeOf(assertions) > 0,
      h3: 'Raw data',
      div @(class: 'request'):{
        div: expander(rawId),
        pre @(class: 'req-res req-res prettyprint $rawId'): (write(elem.result, 'application/json'))
      }
    }
  })
}


fun processSuite(elem: BATTest) = do {
  var carryClassName = sanitizeName(elem.name default 'Suite')
  var testSuites = elem.result map print($, carryClassName)
  ---
  div @(
    tests: testSuites addBy ($.testsuite.@tests default 0),
    errors: testSuites addBy ($.testsuite.@errors default 0),
    failures: testSuites addBy ($.testsuite.@failures default 0),
    skip: testSuites addBy ($.testsuite.@skip default 0),
    (timestamp: elem.startDate) if elem.startDate? and elem.startDate != null
  ): {
    div @(class: 'lead suiteName'): carryClassName,
    (
      if(sizeOf(testSuites) > 0)
        testSuites
      else
        i: '- Empty -'
    )
  }
}

fun pickNonEmpty(list: Array<Any>): String =
  list match {
    case [] -> '?'
    case [head ~ tail] ->
      head match {
        case x if x is String and sizeOf(trim(x)) > 0 -> x as String
        case x is Number -> x as String
        else -> pickNonEmpty(tail) as String
      }
  }

fun iif(condition: Boolean, a: String) =
  if(condition) a else ''

fun passingClass(elem) =
  if(elem.skip == true)
    'skipped'
  else if(elem.pass == true)
    'passing'
  else if(elem.pass == false)
    'failing'
  else
    ''

fun flatObject<T <: Object>(array: Array<T>) = array reduce (z, carry = {}) -> carry ++ z


fun processTest(elem: BATTest, classname: String = '', root: Boolean = false) =
  if(elem.prefix == "SUITE" and root )processSuite(elem)
  else do {
      var package = pickNonEmpty([elem.fileName, elem.name, 'STEP'])
      var steps = elem.result map print($, classname ++ '.' ++ sanitizeName(package))
      var id = getSeqId()
      ---
        div @(
                class: 'test $(passingClass(elem)) $id $(iif(elem.pass == false, "open"))',
                name: elem.name,
                tests: sizeOf(elem.result),
                errors: 0,
                failures: elem.result count ($.pass == false),
                skip: elem.result count ($.skip == true),
                (timestamp: elem.startDate) if elem.startDate? and elem.startDate != null
              ): {
                div @(class: 'test-name $(iif(elem.pass == false, "open"))', onclick: 'ToggleClass("$id")'): {
                  (treeArrow),
                  (bullet),
                  span: elem.name
                },
                (div @(class: 'test-steps'): {(steps)}) if sizeOf(steps) > 0
              }
  }

fun assertionElem(elem: BATPassCapable<Any>, class, label, open = false) =
  using(id = getSeqId()) {
    div @(class: 'docs-example $id $(iif(open, "open"))'): {
      span @(class: 'state $class'): label,
      span @(class: 'assertion-message clickable'): {
        span @(onclick: 'ToggleClass("$id")'): elem.name,
        span: expander(id, true)
      },
    },
    div @(class: 'highlight $id toggable $(iif(open, "open"))'):
      pre @(class: 'code prettyprint'): write(elem.result, 'application/json')
  }

fun print(elem: Any, classname: String = '', root: Boolean = false) =
  elem match {
    case passCapable is BATPassCapable<Any> ->
      passCapable.kind match {
        case "TEST"  -> processTest(passCapable as BATTest, classname, root)
        case "HTTP"  -> processHttp(passCapable as BATHttpStep, classname)
        case "SUITE" -> processSuite(passCapable as BATTest)
        case "LOOPABLE" -> processTest(passCapable as BATTest, classname)
        case "StoreValue" -> {}
        case "Execute" -> {}
        case "Assertion" ->
          if (passCapable.pass == true)
            assertionElem(elem, 'state-open', 'Passed', false)
          else
            assertionElem(elem, 'state-closed', 'Failed', true)
        case "RuntimeException" ->
          itemWithContent({
            label: "Runtime error",
            class: passingClass(elem),
            content: assertionElem(elem, 'state-closed', 'Error', true)
          })
        else ->
          if(DEBUG)
            using(yy = log('No match for pass capable ', elem)) {}
          else
            {}
      }
    else ->
      if(DEBUG)
        using(yy = log('No match for', elem)) {}
      else
        {}
  }

fun count<T>(array: Array<T>, cb: (T) -> Boolean): Number =
  array reduce (item, carry = 0) -> if(cb(item)) carry + 1 else carry

fun addBy<T>(array: Array<T>, cb: (T) -> Number): Number =
  array reduce (item, carry = 0) -> cb(item) + carry