import * from bat::Types
import DEBUG from bat::core::Console

output application/xml

fun sanitizeName(name: String): String =
  (if(name == '') 'STEP' else name)
    replace /([^A-Za-z0-9\s])/ with ''
    replace /(\s)/ with '_'
    replace /(_+)/ with '_'

fun processHttp(elem: BATPassCapable<Any>, classname: String = '') =
  using(time = (elem.time default 0.0) * 0.001)
  using(assertions = elem.assertions map print($, classname ++ '.' ++ sanitizeName(elem.name)))
    testcase @(
      name: elem.name,
      time: time,
      classname: classname,
      status: if(elem.pass) "PASSED" else "FAILED",
      assertions: sizeOf(assertions),
      (timestamp: elem.startDate) if elem.startDate? and elem.startDate != null
    ):
      if (elem.skip == true)
        skipped: null
      else {
        ( assertions ),
        "system-out": write(elem.result, 'application/json') as CData
      }

fun processSuite(elem) = do {
  var carryClassName = sanitizeName(payload.name default 'Suite')
  var testSuites = elem.result map print($, carryClassName)
  ---
  testsuites @(
    name: carryClassName,
    tests: testSuites addBy ($.testsuite.@tests default 0),
    errors: testSuites addBy ($.testsuite.@errors default 0),
    failures: testSuites addBy ($.testsuite.@failures default 0),
    skip: testSuites addBy ($.testsuite.@skip default 0),
    hostname: "localhost",
    (timestamp: elem.startDate) if elem.startDate? and elem.startDate != null
  ): {
    (testSuites),
    "system-out": "" as CData,
    "system-err": "" as CData
  }
}

fun pickNonEmpty(list: Array<Any>): String =
  list match {
    case [] -> '?'
    case [head ~ tail] ->
      head match {
        case x if x is String and sizeOf(trim(x)) > 0 -> x
        case x is Number -> x as String
        else -> pickNonEmpty(tail)
      }
  }

fun processTest(elem: BATTest, classname: String = '') =
  using(package = pickNonEmpty([elem.fileName, elem.name, 'STEP'])
                        replace /(\.dwl)/ with ''
                        replace /(\.)/ with '-'
                        replace /(\\|\/)/ with '.')
    using(steps = elem.result map print($, classname ++ '.' ++ sanitizeName(package)))
      testsuite @(
        name: elem.name,
        tests: sizeOf(elem.result),
        errors: 0,
        failures: elem.result count ($.pass == false),
        skip: elem.result count ($.skip == true),
        (timestamp: elem.startDate) if elem.startDate? and elem.startDate != null
      ): if (elem.skip == true)
           skipped: null
         else {
           (steps)
         }

fun print(elem, classname: String = '') =
  elem match {
    case passCapable if passCapable is BATPassCapable<Any> ->
      passCapable.kind match {
        case "TEST"  ->
          if(passCapable.prefix == "SUITE")
            processSuite(passCapable)
          else
            processTest(passCapable, classname)
        case "HTTP"  -> processHttp(passCapable, classname)
        case "LOOPABLE" -> processTest(passCapable, classname)
        case "StoreValue" -> {}
        case "Execute" -> {}
        case "Assertion" ->
          if (passCapable.pass)
            passed @(message: passCapable.name , "type": passCapable.kind): write(passCapable.result, 'application/json') as CData
          else
            failure @(message: passCapable.name , "type": passCapable.kind): write(passCapable.result, 'application/json') as CData
        case "RuntimeException" ->
          error @(message: passCapable.name, "type": passCapable.result.kind): (passCapable.result.location default '') as CData
        else ->
          if(DEBUG)
            using(yy = log('No match for pass capable ', elem)) {}
          else
            {}
      }
    else ->
      if(DEBUG)
        using(yy = log('No match for', elem)) {}
      else
        {}
  }

fun count<T>(array: Array<T>, cb: (T) -> Boolean): Number =
  array reduce (item, carry = 0) -> if(cb(item)) carry + 1 else carry

fun addBy<T>(array: Array<T>, cb: (T) -> Number): Number =
  array reduce (item, carry = 0) -> cb(item) + carry
---
print(payload)