import compose as URL from dw::core::URL

import * from dw::http::Client
import HttpClientResult from dw::http::Types

import * from bat::Types
import * from bat::core::Executors
import println, printPassCapable, deletePreviousLine, DEBUG from bat::core::Console
import mergeWith from dw::core::Objects
import some from dw::core::Arrays

fun sanitizeName(name: String) = name replace /([\/\]\[|*])/ with '-'

fun stringInterpolation(parts: Array<String>, interpolation: Array<String>): String =
  parts zip interpolation map ($[0] ++ $[1]) joinBy '' ++ parts[-1]

fun wrappedRequest(method: String, url: String, config: HttpCustomOptions): BATHttpStep =
  using(name = config.description default "$method $url")
  // using(preMessage = println("\n  " ++ name ++ " (loading...)", "NORMAL"))
  using(res = request(method, url, config))
  // using(postMessage = log(deletePreviousLine(), ""))
  using(message = if(res.err) println(res.message default "Error during HTTP request", false) else null) //if(res.err) println(res.message, false) else "")
  printPassCapable(
    {
      kind: "HTTP",
      name: name,
      pass: not (res.err default false),
      softFail: false,
      result: res,
      assertions: [] as Array<BATAssertion>,
      time: res.timers.total,
      prefix: method,
      startDate: mockedNow()
    } as BATHttpStep
  )

fun computePass<T>(items: Array<T>): Boolean =
  items reduce (actual, prev = true) ->
    if (actual.kind?)
      actual match {
        case is bat::Mutable::HashMapSetOperation<Any> ->
          prev
        case x if actual.pass? and actual.pass is Boolean ->
          actual.pass and prev
        else -> false
      }
    else
      false



fun executeStatement<X>(fn: (X) -> Any | Boolean, value: X): BATAssertion = do {
  var result = safeExecution(fn, value)
  ---
  result match {
    case x is BATAssertion ->
      if(x.kind == "RuntimeException")
        x // Because safeExecution already prints the RuntimeExceptions
      else
        printPassCapable(x)
    case x is bat::Mutable::HashMapSetOperation<Any> -> printPassCapable(transformSetValue(x))
    else ->
      printPassCapable({
        kind: "Execute",
        pass: true,
        name: "Statement execution",
        softFail: true,
        result: result
      } as BATAssertion)
  }
}

fun executeStatements<X>(value: X, statements: Array<(result: X) -> Any>): Array<BATAssertion> =
  statements map executeStep0(() -> executeStatement($, value), 'Execute')

fun executeAssertions<X>(value: X, assertions: Array<(result: X) -> BATAssertion | Boolean>): Array<BATAssertion> =
  assertions map executeStep0(() -> executeAssertion($, value), 'Assertion')

fun executeAssertion<X>(fn: (X) -> BATAssertion | Boolean, value: X): BATAssertion = do {
  var result = safeExecution(fn, value)
  ---
  result match {
    // add error recovery test
    case pass is Boolean -> printPassCapable({
                         kind: "Assertion",
                         pass: pass,
                         name: "(Boolean assertion)",
                         softFail: false,
                         result: {
                           kind: "Boolean",
                           givenValue: pass,
                           expectedValue: true
                         }
                       } as BATAssertion)
    case x is BATAssertion ->
      if(x.kind == "RuntimeException")
        x // Because safeExecution already prints the RuntimeExceptions
      else
        printPassCapable(x)
    else -> printPassCapable(result as BATAssertion)
  }
}

fun executeAssertion(name: String = "(Boolean assertion)", fn: () -> BATAssertion | Boolean): BATAssertion =
  using (result = safeExecution(fn))
    ((result match {
        // add error recovery test
        case pass is Boolean -> {
                             kind: "Assertion",
                             pass: pass,
                             name: name,
                             softFail: false,
                             result: {
                               kind: "Boolean",
                               givenValue: pass,
                               expectedValue: true
                             }
                           } as BATAssertion
        else -> result as BATAssertion
      }) mergeWith { name: name }) as BATAssertion


fun createRequest(parts: Array<String>, interpolation: Array<String | Number>, method: String) =
  (configDelegate: (Null, Null) -> HttpCustomOptions) -> // This is for the infix `with`
    wrappedRequest(method, stringInterpolation(parts, interpolation map $ as String), configDelegate(null, null))


fun createNamedBlock(prefix: String, name: String, opts: { softFail?: Boolean, skip?: Boolean, kind?: String, metadata?: { _?: Any } } = {}): BATTest = {
  kind: opts.kind default "TEST",
  name: name,
  result: [],
  pass: true,
  (softFail: opts.softFail default false) if opts.softFail?,
  (skip: opts.skip default false) if opts.skip?,
  prefix: prefix,
  (metadata: opts.metadata default {}) if opts.metadata? and opts.metadata is Object,
  startDate: mockedNow()
} as BATTest

fun mockedNow() = if(DEBUG) |2017-10-10T15:10:29| else now()

fun describeSequence<X>(sequence: BATTest, statements: Array<() -> X>) = // This signature should be Array<() -> Array<T> | T>
  executeStep0(() ->
    if(sequence.skip == true)
      printPassCapable(sequence)
    else if(sequence.kind == "LOOPABLE")
      (sequence.prefix match {
        case "REPEAT" -> do {
          var iterations: Number = sequence.metadata.batLoopable.times as Number
          var msg = println(sequence.name)
          var iterationsResults = 1 to iterations map describeSequence(createNamedBlock("ITERATION", $ as String), statements)
          var results = flatList(iterationsResults)
          var result = sequence.result ++ results
          ---
          sequence mergeWith {
            result: result,
            pass: computePass(result)
          }
        }
      }) as BATTest
    else do {
      var msg = printPassCapable(sequence)
      var results = flatList(runSequence(statements))
      var result = sequence.result ++ results
      ---
      sequence mergeWith {
        result: result,
        pass: computePass(result)
      }
    } as BATTest
  , sequence.kind)

fun flatList(items) = items match {
  case [] -> []
  case [head ~ tail] ->
    head match {
        case x is Array -> flatList(head) ++ flatList(tail)
        else -> [head ~ flatList(tail)]
    }
}
