import HttpClientResult from dw::http::Types
import * from bat::core::AssertionHelpers
import bat from bat::core::AssertionHelpers
import * from bat::Types
import fail as raiseError from dw::Runtime

ns bat http://mulesoft.com/ns/bat

import executeAssertions, computePass, executeStatements, executeAssertion from bat::core::Helpers

// ASSERTIONS ==========================================================================================================

/**
*
* Function to perform multiple assertions over the result of an Http test step.
* Meant to be used after HTTP functions like `GET`, `POST`, etc.
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | batResult |
* | assertions |
* |===
*
* === Example
*
* This example shows how the `assert` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from bat::BDD
* import * from bat::Assertions
* ---
* describe `Deck of cards` in [
*   // First we get a new deck of cards
*   GET `http://deckofcardsapi.com/api/deck/new/shuffle/?deck_count=1` with {} assert [
*     $.response.status mustEqual 200, // <--- Then a status assertion
*     $.response.mime mustEqual "application/json", // <--- And a MIME type assertion
*     $.response.body.remaining mustEqual 52 // <--- And finally a boolean assertion
*   ]
* ]
*
* ----
*
**/
fun assert<T>(batResult: BATPassCapable<T>, assertions: Array<(result: T) -> BATAssertion | Boolean>) =
  if (batResult.pass == false)
    batResult
  else do {
    var newAssertions = executeAssertions(batResult.result, assertions)
    var newResult = (batResult - "assertions") ++ {
      assertions: (batResult.assertions default []) ++ newAssertions
    }
    ---
    (newResult - "pass") ++ {
      pass: computePass(newResult.assertions)
    }
  }

/**
* Creates a single named assertions. Meant to be used inside `assert [ ... ]` function.
* Useful to add a meaningful name to the assertion, so it displays useful information in the reporters.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | assertion | a function that returns a `BATAssertion` or a `Boolean`
* | name | display name of the assertion. Mostly used in the reporters
* |===
*
* === Example
*
* This example shows how the `assert` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from bat::BDD
* import * from bat::Assertions
* fun validateStatus(response): Boolean = response.status mustEqual 200
* ---
* describe `Deck of cards` in [
*   // First we get a new deck of cards
*   GET `http://deckofcardsapi.com/api/deck/new/shuffle/?deck_count=1` with {} assert [
*     assert(validateStatus($.response), "Status code must be OK"),
*     $.response.body.remaining mustEqual 52 // <--- And finally a boolean assertion
*   ]
* ]
*
* ----
**/
fun assert(assertion: () -> (BATAssertion | Boolean), name: String) = do {
  executeAssertion(name, assertion)
}

/**
*
* Creates a single named assertions. Meant to be used inside `assert [ ... ]` function.
* Useful to add a meaningful name to the assertion, so it displays useful information in the reporters.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | assertion | a Boolean or BATAssertion value
* | name | display name of the assertion. Mostly used in the reporters
* |===
*
* === Example
*
* This example shows how the `assert` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from bat::BDD
* import * from bat::Assertions
* ---
* describe `Deck of cards` in [
*   // First we get a new deck of cards
*   GET `http://deckofcardsapi.com/api/deck/new/shuffle/?deck_count=1` with {} assert [
*     assert($.response.status == 200, "Status must be OK"), // <--- Then a status assertion
*     $.response.mime mustEqual "application/json", // <--- And a MIME type assertion
*     $.response.body.remaining mustEqual 52 // <--- And finally a boolean assertion
*   ]
* ]
*
* ----
**/
fun assert(assertion: BATAssertion | Boolean, name: String) = do {
  executeAssertion(name, () -> assertion)
}


// EXECUTE =============================================================================================================
/**
*
* Function to execute statements given the result of an HTTP Test block.
* Useful when you want to save a value of the response in a variable or log something you want.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | batResult | an http block
* | statements | the statements to execute
* |===
*
* === Example
*
* This example shows how the `execute` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from bat::BDD
* import * from bat::Assertions
*
* var context = bat::Mutable::HashMap() // <--- First, the HashMap
* ---
* describe `Deck of cards` in [
*   // Then we get a new deck of cards
*   GET `http://deckofcardsapi.com/api/deck/new/shuffle/?deck_count=1` with {} assert [
*     $.response.status mustEqual 200, // <--- Then a status assertion
*     $.response.mime mustEqual "application/json", // <--- And a MIME type assertion
*     $.response.body.remaining mustEqual 52 // <--- And a boolean assertion
*   ] execute [
*     context.set('deck_id', $.response.body.deck_id), // <--- Setting deck_id
*     log($.response) // <--- Then we’ll log the response
*   ]
* ]
*
*
* ----
*
**/
fun execute<T>(batResult: BATPassCapable<T>, statements: Array<(result: T) -> Any>) = do {
  var results = executeStatements(batResult.result, statements)
  var newResult = (batResult - "assertions") ++ {
    assertions: (batResult.assertions default []) ++ results
  }
  ---
  (newResult - "pass") ++ {
    pass: computePass(newResult.assertions)
  }
}

// MATCHERS ============================================================================================================

/**
*
* An assertion helper function that will check if element `a` is similar to element `b`.
* For example you can compare the `Number` 52 with the `String` "52", and it will return true.
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | a | given value
* | b | expected value
* |===
*
* === Example
*
* This example shows how the `mustBeLike` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from bat::BDD
* import * from bat::Assertions
* ---
* describe `Deck of cards` in [
*   // First we get a new deck of cards
*   GET `http://deckofcardsapi.com/api/deck/new/shuffle/?deck_count=1` with {} assert [
*     assert($.response.status == 200, "Status must be OK"), // <--- Then a status assertion
*     $.response.mime mustEqual "application/json", // <--- And a MIME type assertion
*     $.response.body.remaining mustBeLike "52" // <--- And finally a boolean assertion
*   ]
* ]
*
* ----
**/
fun mustBeLike<A,B>(a: A, b: B): BATAssertion = do {
  fun privateMatchLike_<T, V>(value: T, comparator: V): Boolean =
    if(value ~= comparator)
      true
    else
      comparator match {
        case a if isOneOf_(a) ->
          (a as Object).bat#items reduce (comparator, acc = false) -> (acc or privateMatchLike_(value, comparator))
        else -> false
      }
  ---
  {
    kind: "Assertion",
    pass:
      if(isEvery_(a))
        computeEvery_((a as Object).bat#items, (a) -> privateMatchLike_(a, b))
      else if(isSome_(a))
        computeSome_((a as Object).bat#items, (a) -> privateMatchLike_(a, b))
      else
        privateMatchLike_(a, b),
    name: "$(matchPrinter_(a)) must be similar to $(matchPrinter_(b))",
    softFail: false,
    result: {
      kind: "Assertion::Similar",
      expectedValue: write(b),
      givenValue: write(a)
    }
  } as BATAssertion
}

/**
*
* Matcher function that will assert that the value `a` is equals to value `b`.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | a | given value
* | b | expected value
* |===
*
* === Example
*
* This example shows how the `mustEqual` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from bat::BDD
* import * from bat::Assertions
* ---
* describe `Deck of cards` in [
*   // First we get a new deck of cards
*   GET `http://deckofcardsapi.com/api/deck/new/shuffle/?deck_count=1` with {} assert [
*     assert($.response.status == 200, "Status must be OK"), // <--- Then a status assertion
*     $.response.mime mustEqual "application/json", // <--- And a MIME type assertion
*     $.response.body.remaining mustEqual 52 // <--- And finally a boolean assertion
*   ]
* ]
*
* ----
**/
fun mustEqual<A, B>(a: A, b: B): BATAssertion = do {
  fun privateMatchEqual_<T, V>(value: T, comparator: V): Boolean =
    if(value == comparator)
      true
    else
      comparator match {
        case a if isOneOf_(a) ->
          (a as Object).bat#items reduce (comparator, acc = false) -> (acc or privateMatchEqual_(value, comparator))
        else -> false
      }
  ---
  {
    kind: "Assertion",
    pass:
      if(isEvery_(a))
        computeEvery_((a as Object).bat#items, (a) -> privateMatchEqual_(a, b))
      else if(isSome_(a))
        computeSome_((a as Object).bat#items, (a) -> privateMatchEqual_(a, b))
      else
        privateMatchEqual_(a, b),
    name: "$(matchPrinter_(a)) must equal $(matchPrinter_(b))",
    softFail: false,
    result: {
      kind: "Assertion::Equal",
      expectedValue: write(b),
      givenValue: write(a)
    }
  } as BATAssertion
}

/**
*
* Matcher function that can be used to match element `a` with the regex or the value `b`.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | a | the value you want to compare
* | b | the exxepected valuer. Can be a regex, an array or an object
* |===
*
* === Example
*
* This example shows how the `mustMatch` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from bat::BDD
* import * from bat::Assertions
* ---
* describe `Deck of cards` in [
*   // First we get a new deck of cards
*   GET `http://deckofcardsapi.com/api/deck/new/shuffle/?deck_count=1` with {} assert [
*     assert($.response.status == 200, "Status must be OK"), // <--- Then a status assertion
*     $.response.mime mustEqual "application/json",
*     $.response.body.remaining mustMatch /^[0-9]*$/
*   ]
* ]
*
* ----
**/
fun mustMatch<T, V>(a: T, b: () -> V): BATAssertion = do {
  fun privateMatch_<T, V>(value: T, comparator: V): Boolean =
    if(value == comparator)
      true
    else comparator match {
      case is Regex ->
        value match {
          case is Number -> value as String contains comparator
          case is String -> value contains comparator
          else -> false
        }
      case a if isOneOf_(a) ->
        (a as Object).bat#items reduce (comparator, acc = false) -> (acc or privateMatch_(value, comparator))
      // partial object matching
      case is Object ->
        if (value is Object)
          comparator pluck $$ reduce (key, acc = true) -> (acc and privateMatch_(value[key], comparator[key]))
        else
          false
      else -> false
    }
  ---
  {
    kind: "Assertion",
    pass:
      if(isEvery_(a))
        computeEvery_((a as Object).bat#items, (a) -> privateMatch_(a, b()))
      else if(isSome_(a))
        computeSome_((a as Object).bat#items, (a) -> privateMatch_(a, b()))
      else
        privateMatch_(a, b()),
    name: "$(matchPrinter_(a)) must match $(matchPrinter_(b()) as String)",
    softFail: false,
    result: {
      kind: "Assertion::Matcher",
      expectedValue: write(b()),
      givenValue: write(a)
    }
  } as BATAssertion
}

/**
*
* Helper matcher to compare the given value with an array of elements.
* This function will check that the given value is at least equals to one of the elements in the array
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | a | the array of expected elements
* |===
*
* === Example
*
* This example shows how the `oneOf` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from bat::BDD
* import * from bat::Assertions
* ---
* describe `Deck of cards` in [
*   // First we get a new deck of cards
*   GET `http://deckofcardsapi.com/api/deck/new/shuffle/?deck_count=1` with {} assert [
*     assert($.response.status == 200, "Status must be OK"), // <--- Then a status assertion
*     $.response.mime mustEqual "application/json",
*     $.response.body.remaining mustMatch oneOf([50,51,52,53])
*   ]
* ]
*
* ----
**/
fun oneOf<T>(a: Array<T>): MatchesOneOf<T> = {
  bat#matchesOneOf: true,
  bat#items: a
}

fun some<T>(a: Array<T>): EveryItem<T> = {
  bat#everyItem: false,
  bat#items: a
}

fun some(a: Null): EveryItem<Nothing> = {
  bat#everyItem: false,
  bat#items: []
}

/**
*
* Helper matcher to compare the given value with an array of elements.
* This function will check that the given value is equals to every element in the array
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | a | the array of expected elements
* |===
*
* === Example
*
* This example shows how the `every` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from bat::BDD
* import * from bat::Assertions
* ---
* describe `Deck of cards` in [
*   // First we get a new deck of cards
*   GET `http://deckofcardsapi.com/api/deck/new/shuffle/?deck_count=1` with {} assert [
*     assert($.response.status == 200, "Status must be OK"), // <--- Then a status assertion
*     $.response.mime mustEqual "application/json",
*     $.response.body.remaining mustMatch every([52])
*   ]
* ]
*
* ----
**/
fun every<T>(a: Array<T>): EveryItem<T> = {
  bat#everyItem: true,
  bat#items: a
}

fun every(a: Null): EveryItem<Nothing> = {
  bat#everyItem: true,
  bat#items: []
}






