import compose as URL from dw::core::URL

import * from dw::http::Client
import HttpClientResult from dw::http::Types
import envVars from dw::System
import * from dw::util::Values
import * from bat::Types
import * from bat::core::Executors
import println, printPassCapable, deletePreviousLine, DEBUG from bat::core::Console
import mergeWith from dw::core::Objects
import some from dw::core::Arrays
import props from dw::Runtime

fun sanitizeName(name: String) = name replace /([\/\]\[|*])/ with '-'

fun stringInterpolation(parts: Array<String>, interpolation: Array<String>): String =
  parts zip interpolation map ($[0] ++ $[1]) joinBy '' ++ parts[-1]

var PROXY_MODE = envVars().PROXY_MODE? and ((envVars().PROXY_MODE default "") contains "true")
var PROXY_URL = envVars().PROXY_URL default ''
var PROXY_REFERER = envVars().PROXY_REFERER default ''

fun wrapProxyConfig(options: HttpCustomOptions) = do {
    var referer = { "Referer": PROXY_REFERER}
    var newHeaders = options.headers mergeWith referer
    var newOptions = options mergeWith {"headers" : newHeaders}

    ---

    newOptions
}

fun wrappedRequest(method: String, url: String, config: HttpCustomOptions): BATHttpStep = do {
    var proxyEnabled = PROXY_MODE != null and PROXY_URL != null and PROXY_REFERER != null and PROXY_URL != '' and PROXY_REFERER != ''
    var name = (config.description default "$method $url") as String
    var effectiveUrl = if(proxyEnabled == true) PROXY_URL ++ '/' ++ url else url
    var effectiveConf: HttpCustomOptions = if(proxyEnabled == true) wrapProxyConfig(config) else config
    var res = request(method, effectiveUrl, effectiveConf)
    var message = if(res.err) println(res.message default "Error during HTTP request", false) else null
    ---
      printPassCapable(
        {
          kind: "HTTP",
          name: name,
          pass: not (res.err default false),
          softFail: false,
          result: res,
          assertions: [] as Array<BATAssertion>,
          time: res.timers.total,
          prefix: method,
          startDate: mockedNow()
        }
      )
}

fun computePass<T>(items: Array<T>): Boolean =
  (items as Array<Object>) reduce (actual, prev = true) ->
    if ((actual as Object).kind?)
      actual match {
        case is bat::Mutable::HashMapSetOperation<Any> ->
          prev
        case x  if actual.pass? and (actual as Object).pass is Boolean ->
          actual.pass and prev
        else -> false
      }
    else
      false



fun executeStatement<X>(fn: (X) -> Any | Boolean, value: X): BATAssertion = do {
  var result = safeExecution(fn, value)
  ---
  result match {
    case x is BATAssertion ->
      if(x.kind == "RuntimeException")
        x // Because safeExecution already prints the RuntimeExceptions
      else
        printPassCapable(x)
    case x is bat::Mutable::HashMapSetOperation<Any> -> printPassCapable(transformSetValue(x))
    else ->
      printPassCapable({
        kind: "Execute",
        pass: true,
        name: "Statement execution",
        softFail: true,
        result: result
      } as BATAssertion)
  }
}

fun executeStatements<X>(value: X, statements: Array<(result: X) -> Any>): Array<BATAssertion> =
  statements map executeStep0(() -> executeStatement($, value), 'Execute')

fun executeAssertions<X>(value: X, assertions: Array<(result: X) -> BATAssertion | Boolean>): Array<BATAssertion> =
  assertions map executeStep0(() -> executeAssertion($, value), 'Assertion')

fun executeAssertion<X>(fn: (X) -> BATAssertion | Boolean, value: X): BATAssertion = do {
  var result = safeExecution(fn, value)
  ---
  result match {
    // add error recovery test
    case pass is Boolean -> printPassCapable({
                         kind: "Assertion",
                         pass: pass,
                         name: "(Boolean assertion)",
                         softFail: false,
                         result: {
                           kind: "Boolean",
                           givenValue: pass,
                           expectedValue: true
                         }
                       } as BATAssertion)
    case x is BATAssertion ->
      if(x.kind == "RuntimeException")
        x // Because safeExecution already prints the RuntimeExceptions
      else
        printPassCapable(x)
    case x is bat::Mutable::HashMapSetOperation<Any> -> printPassCapable(transformSetValue(x))
    else -> printPassCapable(result as BATAssertion)
  }
}

fun executeAssertion(name: String = "(Boolean assertion)", fn: () -> BATAssertion | Boolean): BATAssertion =
  using (result = safeExecution(fn))
    ((result match {
        // add error recovery test
        case pass is Boolean -> {
                             kind: "Assertion",
                             pass: pass,
                             name: name,
                             softFail: false,
                             result: {
                               kind: "Boolean",
                               givenValue: pass,
                               expectedValue: true
                             }
                           } as BATAssertion
        else -> result as BATAssertion
      }) mergeWith { name: name }) as BATAssertion


fun createRequest(parts: Array<String>, interpolation: Array<String | Number>, method: String) =
  (configDelegate: (Null, Null) -> HttpCustomOptions) -> // This is for the infix `with`
    wrappedRequest(method, stringInterpolation(parts, interpolation map $ as String), configDelegate(null, null))


fun createNamedBlock(prefix: String, name: String, opts: { softFail?: Boolean, skip?: Boolean, kind?: String, metadata?: { _?: Any } } = {}): BATTest = {
  kind: opts.kind default "TEST",
  name: name,
  result: [],
  pass: true,
  (softFail: opts.softFail default false) if opts.softFail?,
  (skip: opts.skip default false) if opts.skip?,
  prefix: prefix,
  (metadata: opts.metadata default {}) if opts.metadata? and opts.metadata is Object,
  startDate: mockedNow()
} as BATTest

fun mockedNow() = if(DEBUG) |2017-10-10T15:10:29| else now()

fun describeSequence<X>(sequence: BATTest, statements: Array<() -> X>) = // This signature should be Array<() -> Array<T> | T>
  executeStep0(() ->
    if(sequence.skip == true)
      printPassCapable(sequence)
    else if(sequence.kind == "LOOPABLE")
      (sequence.prefix match {
        case "REPEAT" -> do {
          var iterations: Number = sequence.metadata.batLoopable.times as Number
          var msg = println(sequence.name)
          var iterationsResults = 1 to iterations map describeSequence(createNamedBlock("ITERATION", $ as String), statements)
          var results = flatList(iterationsResults)
          var result = sequence.result ++ results
          ---
          sequence mergeWith {
            result: result,
            pass: computePass(result)
          }
        }
      }) as BATTest
    else do {
      var msg = printPassCapable(sequence)
      var results = flatList(runSequence(statements))
      var result = sequence.result ++ results
      ---
      sequence mergeWith {
        result: result,
        pass: computePass(result)
      }
    } as BATTest
  , sequence.kind)

fun flatList(items) = items match {
  case [] -> []
  case [head ~ tail] ->
    head match {
        case x is Array -> flatList(head) ++ flatList(tail)
        else -> [head ~ flatList(tail)]
    }
}

fun filterMetadata(value: Any, allowedList: Array<String>) = do {
   value match {
       case theObject is Object ->
            (theObject mapObject (
                (value, key, index) -> {
                        (key): filterMetadata(value, allowedList)
                    }
                )
            ) as Object {(class: theObject.^class) if (allowedList contains theObject.^class)}
       case is Array -> ($ map filterMetadata($, allowedList)) as Array {}
       case is String -> $ as String {}
       case is Boolean -> $ as Boolean {}
       case is Number -> $ as Number {}
       case is Date -> $ as Date {}
       case is DateTime -> $ as DateTime {}
       case is LocalDateTime -> $ as LocalDateTime {}
       case is Time -> $ as Time {}
       case is LocalTime -> $ as LocalTime {}
       case is Binary -> $ as Binary {}
       case is Regex -> $ as Regex {}
       case is Namespace -> $ as Namespace {}
       else -> $ as Any {}
   }
}

fun processSuite(elem) = do {
  flatten(elem.result map getHttpElements($, elem.name, elem.name))
}

fun processTest(elem: BATPassCapable<Any>, suite: String, classname: String = '') = do {
  var package = elem.name
  var head = if(classname == '') '' else (classname ++ '/')
  ---
  if (elem.prefix == "SUITE")
    processSuite(elem)
  else
    flatten(elem.result map getHttpElements($, suite, head ++ sanitizeName(package)))
}

fun getHttpElements(elem, suite: String, classname: String = '')=
          elem.kind default "" match {
            case "TEST"  -> processTest(elem, suite, classname)
            case "HTTP"  -> elem
            case "LOOPABLE" -> processTest(elem, suite, classname)
            else -> []
          }

fun getTimeFromHttp(httpCall, payload) = do {
   if (httpCall.startDate matches /^(\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2})$/)
     httpCall.startDate
   else
     httpCall.startDate as Number >> "milliseconds" as DateTime {format: "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'"} default payload.startDate as Number >> "milliseconds" as DateTime {format: "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'"} default now() as DateTime {format: "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'"}
}
