/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.mojo;

import static java.lang.String.format;
import static java.nio.file.Files.exists;
import static java.nio.file.Files.write;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.maven.plugins.annotations.LifecyclePhase.PROCESS_CLASSES;

import org.apache.maven.plugin.MojoFailureException;
import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.deployment.persistence.MulePluginModelJsonSerializer;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.List;

import com.google.gson.GsonBuilder;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;

@Mojo(name = "ExportDataweaveResources", defaultPhase = PROCESS_CLASSES)
public class ExportDataweaveResourcesMojo extends BaseRestCommonsMojo {

  private static final String DATAWEAVE_SUBFOLDER = "dw";
  private static final String DATAWEAVE_EXTENSION = ".dwl";

  @Parameter(property = "rest.connector.commons.shortName", defaultValue = "${shortName}")
  private String shortName;

  @Parameter(property = "rest.connector.commons.skipExport", defaultValue = "false")
  private boolean skipExport;

  @Override
  protected void doExecute() throws MojoExecutionException {

    if (skipExport) {
      getLog().warn("Will not export DW resources");
      return;
    }

    if (isBlank(shortName)) {
      getLog()
          .warn("Connector shortName not found, will not export DW resources. Please specify the 'rest.connector.commons.shortName'");
      return;
    }

    if (project.getResources().isEmpty()) {
      getLog().info("Resources directory not found, will not export DW resources.");
      return;
    }

    String projectBaseDir = project.getBasedir().toString();
    project.getResources().forEach(resource -> {
      try {
        Files.walk(Paths.get(resource.getDirectory()))
            .filter(Files::isRegularFile)
            .filter(file -> file.getFileName().toString().endsWith(DATAWEAVE_EXTENSION))
            .filter(file -> !file.startsWith(Paths.get(resource.getDirectory(), shortName, DATAWEAVE_SUBFOLDER)))
            .forEach(file -> getLog().warn(format("Dataweave file: '%s' will not be exported because it is not placed in: '%s'",
                                                  Paths.get(projectBaseDir).relativize(file), Paths.get(projectBaseDir)
                                                      .relativize(Paths.get(resource.getDirectory(), shortName,
                                                                            DATAWEAVE_SUBFOLDER)))));
      } catch (IOException e) {
        getLog().warn(format("Failed to check for existing dataweave files in: '%s'",
                             Paths.get(projectBaseDir).relativize(Paths.get(resource.getDirectory()))));
      }
    });

    if (project.getResources().stream()
        .noneMatch(resource -> exists(Paths.get(resource.getDirectory(), shortName, DATAWEAVE_SUBFOLDER)))) {
      project.getResources().forEach(resource -> getLog()
          .info(format("Resources directory: '%s' does not exists, will not export DW resources.",
                       Paths.get(projectBaseDir)
                           .relativize(Paths.get(resource.getDirectory(), shortName, DATAWEAVE_SUBFOLDER)))));
      return;
    }

    String projectOutputDirectory = project.getBuild().getOutputDirectory();
    final Path muleArtifactPath = Paths.get(projectOutputDirectory, META_INF, MULE_ARTIFACT, MULE_ARTIFACT_JSON);
    String muleArtifact = getMuleArtifact(muleArtifactPath);
    MulePluginModel mulePluginModel = new MulePluginModelJsonSerializer().deserialize(muleArtifact);
    List<String> exportedResources =
        (List<String>) mulePluginModel.getClassLoaderModelLoaderDescriptor().getAttributes().get("exportedResources");
    try {
      Path dwResourcesRelativePath = Paths.get(shortName, DATAWEAVE_SUBFOLDER);
      Path muleArtifactRelativePath = Paths.get(projectOutputDirectory).relativize(muleArtifactPath);
      getLog().info(format("Exporting resources under: '%s' into the '%s'", dwResourcesRelativePath, muleArtifactRelativePath));
      Files.walk(Paths.get(projectOutputDirectory, dwResourcesRelativePath.toString()))
          .filter(Files::isRegularFile)
          .map(file -> Paths.get(projectOutputDirectory).relativize(file))
          .forEach(file -> {
            getLog().info(format("Exporting: '%s'", file));
            exportedResources.add(file.toString());
          });
      write(muleArtifactPath, new GsonBuilder().setPrettyPrinting().create().toJson(mulePluginModel).getBytes(),
            StandardOpenOption.CREATE);
    } catch (IOException e) {
      throw new MojoExecutionException(format("Failed to write mule artifact file: %s", e.getMessage()));
    }

  }

}
