/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.customization;

import static java.lang.String.format;
import static java.nio.charset.Charset.defaultCharset;
import static org.apache.commons.io.FileUtils.writeByteArrayToFile;

import org.mule.weave.v2.runtime.DataWeaveResult;
import org.mule.weave.v2.runtime.DataWeaveScript;
import org.mule.weave.v2.runtime.DataWeaveScriptingEngine;
import org.mule.weave.v2.runtime.ScriptingBindings;

import com.mulesoft.connectivity.rest.sdk.customization.exception.CustomizationException;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Path;

import org.apache.commons.io.IOUtils;

public class CustomizationService {

  private DataWeaveScriptingEngine dataWeaveScriptingEngine;

  public CustomizationService() {
    this.dataWeaveScriptingEngine = new DataWeaveScriptingEngine();
  }

  public void generateCustomizationOverlay(ConnectorModel connectorModel, Path outputDir,
                                           RestSdkRunConfiguration runConfiguration)
      throws CustomizationException {
    String customizationOverlay = generate(connectorModel);
    writeOutputFile(outputDir.resolve(runConfiguration.getCustomizationOverlayDir()),
                    runConfiguration.getCustomizationOverlayFilename(), customizationOverlay.getBytes(defaultCharset()));
  }

  public String generate(ConnectorModel connectorModel) {
    DataWeaveScript compile = dataWeaveScriptingEngine.compile(
                                                               readScriptFile("scripts/customization-overlay.dwl"),
                                                               new String[] {"payload"});

    ScriptingBindings bindings = new ScriptingBindings()
        .addBinding("payload", connectorModel, "application/java");

    DataWeaveResult dataWeaveResult = compile.write(bindings);

    return dataWeaveResult.getContentAsString();
  }

  private String readScriptFile(String scriptRelativePath) {
    try {
      return IOUtils.toString(this.getClass().getClassLoader().getResourceAsStream(scriptRelativePath),
                              defaultCharset());
    } catch (IOException e) {
      throw new UncheckedIOException("There was an error while reading the DataWeave script", e);
    }
  }

  private File writeOutputFile(Path outputDir, String outputFileName, byte[] content) throws CustomizationException {
    try {
      File outputFile = new File(outputDir.toFile(), outputFileName);
      writeByteArrayToFile(outputFile, content, false);
      return outputFile;
    } catch (IOException e) {
      throw new CustomizationException(format("There was an error while writing file '%s'", outputFileName), e);
    }
  }
}
