/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.configuration.layers;

import static javax.lang.model.element.Modifier.ABSTRACT;
import static javax.lang.model.element.Modifier.PUBLIC;

import com.mulesoft.connectivity.rest.commons.api.error.RestError;
import com.mulesoft.connectivity.rest.commons.api.interception.descriptor.HttpResponseInterceptorDescriptor;
import com.mulesoft.connectivity.rest.commons.api.interception.descriptor.builder.HttpResponseInterceptorDescriptorBuilder;
import com.mulesoft.connectivity.rest.commons.api.interception.descriptor.builder.PreconditionHttpResponseInterceptorDelegateDescriptorBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorInterceptor;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

import java.nio.file.Path;
import java.util.List;
import java.util.function.Consumer;
import java.util.stream.Collectors;

public class SdkConfigBaseLayer extends AbstractSdkConfigLayer {

  private static final String GET_RESPONSE_INTERCEPTOR_DESCRIPTOR = "getResponseInterceptorDescriptor";

  private final TypeName superclass;

  public SdkConfigBaseLayer(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector, String javaClassName,
                            String packageName, TypeName superclass, RestSdkRunConfiguration runConfiguration) {
    super(outputDir, connectorModel, sdkConnector, javaClassName, packageName, runConfiguration);
    this.superclass = superclass;
  }

  public String getJavaClassName() {
    return super.getJavaClassName() + BASE_CLASSNAME_SUFFIX;
  }

  public String getPackage() {
    return super.getPackage() + BASE_PACKAGE_SUFFIX;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    generateConfigClass();
  }

  private void generateConfigClass() throws TemplatingException {
    TypeSpec.Builder configClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(PUBLIC)
            .addModifiers(ABSTRACT)
            .superclass(superclass);

    if (connectorModel.getInterceptors() != null && connectorModel.getInterceptors().size() > 0) {
      addResponseInterceptorDescriptor(configClassBuilder);
    }

    JavaFile.Builder javaFileBuilder = getJavaFileBuilderForClass(configClassBuilder.build(), getPackage());
    writeJavaFile(javaFileBuilder.build());
  }

  private CodeBlock generateInterceptorDescriptor() {
    CodeBlock.Builder methodBody = CodeBlock.builder();

    methodBody.addStatement("$1T.Matcher isRestSdkErrorPrecondition = "
        + "statusCode -> $2T.getErrorByCode(statusCode).isPresent()",
                            PreconditionHttpResponseInterceptorDelegateDescriptorBuilder.class,
                            RestError.class);
    methodBody.addStatement("$1T.Matcher isRestSdkNotErrorPrecondition = "
        + "statusCode -> !$2T.getErrorByCode(statusCode).isPresent()",
                            PreconditionHttpResponseInterceptorDelegateDescriptorBuilder.class,
                            RestError.class);

    methodBody.addStatement("$1T httpResponseInterceptorDescriptorBuilder = $2T.builder()",
                            HttpResponseInterceptorDescriptorBuilder.class,
                            HttpResponseInterceptorDescriptor.class);

    List<ConnectorInterceptor> interceptorsOnError =
        connectorModel.getInterceptors().stream().filter(x -> x.isOnError()).collect(Collectors.toList());
    List<ConnectorInterceptor> interceptorsOnSuccess =
        connectorModel.getInterceptors().stream().filter(x -> !x.isOnError()).collect(Collectors.toList());

    if (interceptorsOnError != null && interceptorsOnError.size() > 0) {
      generateInterceptorDescriptorBlocks(methodBody, interceptorsOnError, "onErrorInterceptorDescriptorConfigurator");
      methodBody
          .addStatement("httpResponseInterceptorDescriptorBuilder.when(isRestSdkErrorPrecondition, onErrorInterceptorDescriptorConfigurator)");
    }

    if (interceptorsOnSuccess != null && interceptorsOnSuccess.size() > 0) {
      generateInterceptorDescriptorBlocks(methodBody, interceptorsOnSuccess, "onSuccessInterceptorDescriptorConfigurator");
      methodBody
          .addStatement("httpResponseInterceptorDescriptorBuilder.when(isRestSdkNotErrorPrecondition, onSuccessInterceptorDescriptorConfigurator)");
    }

    methodBody.addStatement("return httpResponseInterceptorDescriptorBuilder.build()");

    return methodBody.build();
  }

  private void generateInterceptorDescriptorBlocks(CodeBlock.Builder methodBody, List<ConnectorInterceptor> interceptors,
                                                   String variableConfiguratorName) {

    methodBody.add("$1T<$2T> " + variableConfiguratorName + " = b -> b",
                   Consumer.class, PreconditionHttpResponseInterceptorDelegateDescriptorBuilder.class);
    for (ConnectorInterceptor connectorInterceptor : interceptors) {
      methodBody.add(".expression(e -> e");
      if (connectorInterceptor.getExpression() != null) {
        methodBody.add(".matchExpression(\"" + connectorInterceptor.getExpression() + "\")\n");
      }
      if (connectorInterceptor.getReasonPhrase() != null) {
        methodBody.add(".reasonPhraseExpression(\"" + connectorInterceptor.getReasonPhrase() + "\")\n");
      }
      if (connectorInterceptor.getStatusCode() != null) {
        methodBody.add(".statusCodeExpression(\"" + connectorInterceptor.getStatusCode() + "\")\n");
      }
      if (connectorInterceptor.getBody() != null) {
        methodBody.add(".bodyExpression(\"" + connectorInterceptor.getBody() + "\")\n");
      }
      if (connectorInterceptor.getHeaders() != null) {
        methodBody.add(".headersExpression(\"" + connectorInterceptor.getHeaders() + "\")\n");
      }
      methodBody.add(")\n");
    }
    methodBody.add(";");
  }

  private void addResponseInterceptorDescriptor(TypeSpec.Builder configClassBuilder) {
    MethodSpec getInterceptors = MethodSpec.methodBuilder(GET_RESPONSE_INTERCEPTOR_DESCRIPTOR)
        .returns(TypeName.get(HttpResponseInterceptorDescriptor.class))
        .addModifiers(PUBLIC)
        .addAnnotation(Override.class)
        .addCode(generateInterceptorDescriptor())
        .build();

    configClassBuilder.addMethod(getInterceptors);
  }
}
