/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.connection;

import com.mulesoft.connectivity.rest.commons.api.connection.MockedConnectionProvider;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.MockedAuthenticationScheme;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.connection.layers.AbstractSdkConnectionProviderLayer;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.TypeSpec;
import com.squareup.javapoet.AnnotationSpec;
import org.mule.runtime.extension.api.annotation.Alias;

import java.nio.file.Path;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.util.SdkTemplatingUtils.getDisplayNameAnnotation;
import static javax.lang.model.element.Modifier.PUBLIC;
import static org.apache.commons.lang.WordUtils.capitalize;

public class SdkMockedConnectionProvider extends AbstractSdkConnectionProvider {

  private final MockedAuthenticationScheme securityScheme;
  private final SdkMockedConnectionProviderLayer sdkMockedConnectionProviderLayer;

  public SdkMockedConnectionProvider(Path outputDir,
                                     ConnectorModel connectorModel,
                                     MockedAuthenticationScheme securityScheme,
                                     RestSdkRunConfiguration runConfiguration) {
    super(outputDir, connectorModel, runConfiguration);

    this.securityScheme = securityScheme;
    this.sdkMockedConnectionProviderLayer =
        new SdkMockedConnectionProviderLayer(outputDir, connectorModel, securityScheme, runConfiguration);
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    sdkMockedConnectionProviderLayer.applyTemplates();
  }

  @Override
  public TypeName getTypeNameForConfig() {
    return ClassName.get(this.sdkMockedConnectionProviderLayer.getPackage(),
                         this.sdkMockedConnectionProviderLayer.getJavaClassName());
  }

  private class SdkMockedConnectionProviderLayer extends AbstractSdkConnectionProviderLayer {

    private final String MOCKED_JAVADOC =
        "Auto generated mocked connection provider.";
    private final String MOCKED =
        "Mocked";

    public SdkMockedConnectionProviderLayer(Path outputDir,
                                            ConnectorModel connectorModel,
                                            ConnectorSecurityScheme securityScheme,
                                            RestSdkRunConfiguration runConfiguration) {
      super(outputDir, connectorModel, securityScheme, null, runConfiguration);
    }

    @Override
    public void applyTemplates() throws TemplatingException {
      generateMockedConnectionProviderClass();
    }

    @Override
    protected String getPackage() {
      return getConnectionProviderBasePackage();
    }

    @Override
    protected String getJavaClassName() {
      return getJavaUpperCamelNameFromXml(getConnectionProviderXmlName()) + MOCKED + CONNECTION_PROVIDER_CLASSNAME_SUFFIX;
    }

    private void generateMockedConnectionProviderClass() throws TemplatingException {

      TypeSpec.Builder connectionProviderClassBuilder =
          TypeSpec
              .classBuilder(getJavaClassName())
              .addModifiers(PUBLIC)
              .superclass(MockedConnectionProvider.class)
              .addAnnotation(generateAliasAnnotation())
              .addAnnotation(getDisplayNameAnnotation(getConnectionProviderDisplayName()))
              .addJavadoc(MOCKED_JAVADOC);

      writeClassToFile(connectionProviderClassBuilder.build(), getPackage());
    }

    private AnnotationSpec generateAliasAnnotation() {
      return AnnotationSpec
          .builder(Alias.class)
          .addMember(VALUE_MEMBER, "$S", capitalize(getConnectionProviderXmlName() + MOCKED))
          .build();
    }

    private String getConnectionProviderDisplayName() {
      return capitalize((getConnectionProviderXmlName() + " " + MOCKED).replace('-', ' ')) + " Connection Provider";
    }
  }
}
