/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.connection.layers;

import static com.google.common.base.CaseFormat.LOWER_CAMEL;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.util.SdkTemplatingUtils.generateGetter;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.util.SdkTemplatingUtils.getDisplayNameAnnotation;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.util.SdkTemplatingUtils.getSummaryAnnotation;
import static javax.lang.model.element.Modifier.PROTECTED;
import static javax.lang.model.element.Modifier.PUBLIC;
import static org.apache.commons.lang3.StringUtils.defaultIfEmpty;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;

import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.meta.ExpressionSupport;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.sdk.api.annotation.semantics.connectivity.Url;
import org.mule.runtime.extension.api.annotation.param.ParameterGroup;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.http.api.HttpConstants;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.auth.HttpAuthentication;
import org.mule.runtime.http.api.client.proxy.ProxyConfig;

import com.mulesoft.connectivity.rest.commons.api.connection.MandatoryTlsParameterGroup;
import com.mulesoft.connectivity.rest.commons.api.connection.OptionalTlsParameterGroup;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.connection.TlsParameterGroup;
import com.mulesoft.connectivity.rest.commons.api.connection.validation.ConnectionValidationSettings;
import com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.TestConnectionConfig;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.TestConnectionValidationConfig;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.uri.BaseUri;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.connection.authentication.SdkAuthenticationStrategy;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.model.SdkHttpProxyConfig;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkParameter;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.StringJoiner;
import java.util.Collections;

import javax.inject.Inject;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public class SdkConnectionProviderBaseLayer extends AbstractSdkConnectionProviderLayer {

  private final static String JAVA_DOC_BASE_URI = "@return the base uri of the REST API being consumed";

  private static final String CREATE_CONNECTION_METHOD_NAME = "createConnection";
  private static final String CREATE_CONNECTION_HTTP_CLIENT_PARAMETER_NAME = "httpClient";
  private static final String CREATE_CONNECTION_AUTHENTICATION_PARAMETER_NAME = "authentication";
  private static final String CREATE_CONNECTION_QUERY_PARAMS_PARAMETER_NAME = "defaultQueryParams";
  private static final String CREATE_CONNECTION_HEADERS_PARAMETER_NAME = "defaultHeaders";

  private static final String VALIDATE_CONNECTION_METHOD_NAME = "validate";
  private static final String VALIDATE_CONNECTION_CONNECTION_PARAMETER_NAME = "restConnection";
  private static final String VALIDATE_CONNECTION_SETTINGS_VAR_NAME = "settings";

  private static final String EXPRESSION_LANGUAGE_FIELD = "expressionLanguage";
  public static final String BASE_URI_FIELD = "baseUri";
  public static final String TLS_CONFIG_FIELD = "tlsConfig";
  public static final String PROXY_CONFIG_FIELD = "proxyConfig";

  public static final String DEFAULT_VALUE_MEMBER = "defaultValue";

  public static final String BASE_LAYER_JAVADOC =
      "This is the first layer of the connection provider generation gap pattern. "
          + "It contains most of the logic of the connection provider.";

  private final List<SdkParameter> headers;
  private final List<SdkParameter> queryParameters;
  private final SdkHttpProxyConfig httpProxyConfig;

  public SdkConnectionProviderBaseLayer(Path outputDir,
                                        SdkConnector sdkConnector,
                                        ConnectorModel connectorModel,
                                        ConnectorSecurityScheme securityScheme,
                                        SdkAuthenticationStrategy authenticationStrategy,
                                        SdkHttpProxyConfig httpProxyConfig,
                                        RestSdkRunConfiguration runConfiguration) {
    super(outputDir, connectorModel, securityScheme, authenticationStrategy, runConfiguration);

    this.httpProxyConfig = httpProxyConfig;
    headers = buildSdkParameters(securityScheme.getHeaders(), outputDir, connectorModel, sdkConnector);
    queryParameters = buildSdkParameters(securityScheme.getQueryParameters(), outputDir, connectorModel, sdkConnector);

  }

  private List<SdkParameter> buildSdkParameters(
                                                List<com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter> parameters,
                                                Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector) {

    final List<SdkParameter> list = new ArrayList<>();
    for (com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter param : parameters) {
      list.add(new SdkParameter(outputDir, connectorModel, sdkConnector, getJavaClassName(), param, this, runConfiguration));
    }
    return list;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    generateConnectionProviderClass();
  }

  public String getJavaClassName() {
    return getJavaUpperCamelNameFromXml(getConnectionProviderXmlName()) + CONNECTION_PROVIDER_CLASSNAME_SUFFIX
        + BASE_CLASSNAME_SUFFIX;
  }

  public String getPackage() {
    return getConnectionProviderBasePackage() + BASE_PACKAGE_SUFFIX;
  }

  private void generateConnectionProviderClass() throws TemplatingException {
    FieldSpec baseUriField = generateBaseUriField();

    TypeSpec.Builder connectionProviderClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(PUBLIC)
            .superclass(authenticationStrategy.getCommonsBaseClass())
            .addField(baseUriField)
            .addJavadoc(BASE_LAYER_JAVADOC)
            .addMethod(generateGetter(baseUriField, LOWER_CAMEL).addAnnotation(Override.class)
                .addJavadoc(CodeBlock.builder().add(JAVA_DOC_BASE_URI).add("\n").build()).build());

    authenticationStrategy.addDefaultClassMembers(connectionProviderClassBuilder);

    addTls(connectionProviderClassBuilder);
    addCreateConnectionOverrideMethod(authenticationStrategy, connectionProviderClassBuilder);

    addHttpProxyConfig(connectionProviderClassBuilder, httpProxyConfig.getTypeName());

    for (SdkParameter header : headers) {
      addParameterField(authenticationStrategy, connectionProviderClassBuilder, header);
    }

    for (SdkParameter queryParam : queryParameters) {
      addParameterField(authenticationStrategy, connectionProviderClassBuilder, queryParam);
    }

    if (shouldTestConnectivity()) {
      addValidateConnectionMethod(connectionProviderClassBuilder);
    }

    writeClassToFile(connectionProviderClassBuilder.build(), getPackage());
  }

  private FieldSpec generateBaseUriField() {

    FieldSpec.Builder baseUriFieldSpec = FieldSpec
        .builder(String.class, BASE_URI_FIELD, PROTECTED)
        .addAnnotation(getDisplayNameAnnotation("Base Uri"))
        .addAnnotation(getSummaryAnnotation(connectorModel.getBaseUri().getType().getDescription()))
        .addJavadoc(CodeBlock.builder().add(JAVA_DOC_BASE_URI).build());

    if (connectorModel.getBaseUri().isParameterizedBaseUri()) {
      baseUriFieldSpec
          .addAnnotation(Parameter.class)
          .addAnnotation(Url.class);
      if (isNotBlank(connectorModel.getBaseUri().getUri())) {
        baseUriFieldSpec.addAnnotation(buildBaseUriOptionalAnnotation(connectorModel.getBaseUri()));
      }
    } else {
      baseUriFieldSpec.initializer("\"" + connectorModel.getBaseUri().getUri() + "\"");
    }

    return baseUriFieldSpec.build();
  }

  private AnnotationSpec buildBaseUriOptionalAnnotation(BaseUri baseUri) {
    return AnnotationSpec
        .builder(Optional.class)
        .addMember(DEFAULT_VALUE_MEMBER, "$S", connectorModel.getBaseUri().getUri())
        .build();
  }

  private void addTls(TypeSpec.Builder connectionProviderClassBuilder) {
    if (connectorModel.supportsHTTPS()) {
      FieldSpec tlsField = generateTlsField();
      CodeBlock.Builder tlsJavaDocBuilder = CodeBlock.builder();

      if (connectorModel.supportsHTTP()) {
        tlsJavaDocBuilder
            .add("\n{@link $L} that configures TLS and allows to switch between HTTP and HTTPS protocols.\n\n",
                 TlsParameterGroup.class.getSimpleName());
      } else {
        tlsJavaDocBuilder.add("\n{@link $L} that configures TLS for this connection.\n\n",
                              TlsParameterGroup.class.getSimpleName());
      }

      tlsJavaDocBuilder.add("@return an optional {@link $L}", TlsParameterGroup.class.getSimpleName()).build();

      connectionProviderClassBuilder
          .addField(tlsField)
          .addMethod(
                     generateOptionalGetter(tlsField, TlsParameterGroup.class, LOWER_CAMEL)
                         .addAnnotation(Override.class)
                         .addJavadoc(tlsJavaDocBuilder.build())
                         .build());
    }
  }

  private FieldSpec generateTlsField() {
    Class<?> tlsClass = connectorModel.supportsHTTP() ? OptionalTlsParameterGroup.class : MandatoryTlsParameterGroup.class;
    return FieldSpec
        .builder(tlsClass, TLS_CONFIG_FIELD, PROTECTED)
        .addJavadoc(CodeBlock.builder()
            .add("{@link $L} references to a TLS config element. This will enable HTTPS for this config.\n",
                 tlsClass.getSimpleName())
            .build())
        .addAnnotation(AnnotationSpec.builder(ParameterGroup.class).addMember(NAME_MEMBER, "$S", "tls").build())
        .build();
  }

  private void addCreateConnectionOverrideMethod(SdkAuthenticationStrategy authenticationStrategy,
                                                 TypeSpec.Builder connectionProviderClassBuilder) {
    if ((!headers.isEmpty() || !queryParameters.isEmpty()) && authenticationStrategy.canOverrideCreateConnectionMethod()) {
      MethodSpec createConnectionMethod = MethodSpec.methodBuilder(CREATE_CONNECTION_METHOD_NAME)
          .returns(TypeName.get(RestConnection.class))
          .addModifiers(PROTECTED)
          .addAnnotation(Override.class)
          .addParameter(HttpClient.class, CREATE_CONNECTION_HTTP_CLIENT_PARAMETER_NAME)
          .addParameter(HttpAuthentication.class, CREATE_CONNECTION_AUTHENTICATION_PARAMETER_NAME)
          .addParameter(getStringMultiMapTypeName(), CREATE_CONNECTION_QUERY_PARAMS_PARAMETER_NAME)
          .addParameter(getStringMultiMapTypeName(), CREATE_CONNECTION_HEADERS_PARAMETER_NAME)
          .addCode(generateCreateConnectionMethodBody())
          .build();

      connectionProviderClassBuilder.addMethod(createConnectionMethod);
    }
  }

  private TypeName getStringMultiMapTypeName() {
    return ParameterizedTypeName.get(MultiMap.class, String.class, String.class);
  }

  private CodeBlock generateCreateConnectionMethodBody() {
    CodeBlock.Builder methodBody = CodeBlock.builder();

    addCustomParameters(methodBody, queryParameters, CREATE_CONNECTION_QUERY_PARAMS_PARAMETER_NAME);
    addCustomParameters(methodBody, headers, CREATE_CONNECTION_HEADERS_PARAMETER_NAME);

    methodBody.addStatement("return super.createConnection($L, $L, $L, $L)",
                            CREATE_CONNECTION_HTTP_CLIENT_PARAMETER_NAME,
                            CREATE_CONNECTION_AUTHENTICATION_PARAMETER_NAME,
                            CREATE_CONNECTION_QUERY_PARAMS_PARAMETER_NAME,
                            CREATE_CONNECTION_HEADERS_PARAMETER_NAME);

    return methodBody.build();
  }

  private void addCustomParameters(CodeBlock.Builder methodBody, List<SdkParameter> parameters, String parametersParameterName) {
    if (!parameters.isEmpty()) {
      methodBody
          .beginControlFlow("if($L == null)", parametersParameterName)
          .addStatement("$L = new $T.StringMultiMap()", parametersParameterName, MultiMap.class)
          .endControlFlow();

      for (SdkParameter header : parameters) {
        methodBody
            .beginControlFlow("if($T.isNotBlank($L))", RestSdkUtils.class, header.getJavaName())
            .addStatement("$L.put($S, $L)",
                          parametersParameterName,
                          header.getExternalName(),
                          header.getJavaName())
            .endControlFlow();
      }
    }
  }

  private void addHttpProxyConfig(TypeSpec.Builder connectionProviderClassBuilder, TypeName proxyConfigTypeName) {
    FieldSpec fieldSpec = generateHttpProxyConfigField(proxyConfigTypeName);
    connectionProviderClassBuilder.addField(fieldSpec);

    MethodSpec getProxyConfigMethod =
        generateGetter(fieldSpec, LOWER_CAMEL, PROTECTED)
            .returns(ProxyConfig.class)
            .addAnnotation(Override.class)
            .build();

    connectionProviderClassBuilder.addMethod(getProxyConfigMethod);
  }

  private FieldSpec generateHttpProxyConfigField(TypeName proxyConfigTypeName) {
    FieldSpec.Builder proxyConfigSpec = FieldSpec
        .builder(proxyConfigTypeName, PROXY_CONFIG_FIELD, PROTECTED)
        .addAnnotation(Parameter.class)
        .addAnnotation(Optional.class)
        .addAnnotation(AnnotationSpec.builder(Expression.class)
            .addMember(VALUE_MEMBER, "$T.$L", ExpressionSupport.class, NOT_SUPPORTED).build())
        .addAnnotation(getSummaryAnnotation("Reusable configuration element for outbound connections through a proxy"))
        .addAnnotation(AnnotationSpec.builder(Placement.class)
            .addMember("tab", "$S", "Proxy").build())
        .addJavadoc(CodeBlock.builder()
            .add("Reusable configuration element for outbound connections through a proxy. \nA proxy element must define a host name and a port attributes, and optionally can define a username and a password.")
            .build());
    return proxyConfigSpec.build();
  }

  private void addParameterField(SdkAuthenticationStrategy authenticationStrategy,
                                 TypeSpec.Builder connectionProviderClassBuilder,
                                 SdkParameter parameter) {

    FieldSpec.Builder fieldBuilder = authenticationStrategy.getParameterField(parameter);

    fieldBuilder
        .addModifiers(PROTECTED)
        .addJavadoc(generateSdkParameterJavaDoc(parameter));

    connectionProviderClassBuilder.addField(fieldBuilder.build());
  }

  private CodeBlock generateSdkParameterJavaDoc(SdkParameter sdkParameter) {
    return CodeBlock.builder()
        .add("$L\n", defaultIfEmpty(sdkParameter.getDescription(), sdkParameter.getDisplayName())).build();
  }

  private boolean shouldTestConnectivity() {
    return securityScheme.getTestConnectionConfig() != null;
  }

  private void addValidateConnectionMethod(TypeSpec.Builder connectionProviderClassBuilder) {
    FieldSpec expressionLanguageField = FieldSpec
        .builder(ExpressionLanguage.class, EXPRESSION_LANGUAGE_FIELD, PROTECTED)
        .addAnnotation(AnnotationSpec.builder(Inject.class).build())
        .build();

    connectionProviderClassBuilder.addField(expressionLanguageField);

    MethodSpec validateConnectionMethod = MethodSpec.methodBuilder(VALIDATE_CONNECTION_METHOD_NAME)
        .returns(TypeName.get(ConnectionValidationResult.class))
        .addModifiers(PUBLIC)
        .addAnnotation(Override.class)
        .addParameter(RestConnection.class, VALIDATE_CONNECTION_CONNECTION_PARAMETER_NAME)
        .addCode(generateValidateConnectionMethodBody())
        .build();

    connectionProviderClassBuilder.addMethod(validateConnectionMethod);
  }

  private CodeBlock generateValidateConnectionMethodBody() {
    CodeBlock.Builder methodBody = CodeBlock.builder();

    TestConnectionConfig testConnectionConfig = securityScheme.getTestConnectionConfig();

    String path = testConnectionConfig.getPath();
    String method = testConnectionConfig.getMethod() != null ? testConnectionConfig.getMethod().name() : null;
    Set<String> validStatusCodes = testConnectionConfig.getStatusCodeValidationConfig() != null
        ? testConnectionConfig.getStatusCodeValidationConfig().getValidStatusCodes()
        : Collections.emptySet();

    methodBody.add("$1T $2L = $1T.builder($3S, $4L)",
                   ConnectionValidationSettings.class,
                   VALIDATE_CONNECTION_SETTINGS_VAR_NAME,
                   path,
                   EXPRESSION_LANGUAGE_FIELD);

    if (isNotBlank(method)) {
      methodBody.add(".httpMethod($T.$L)", HttpConstants.Method.class, method);
    }

    if (!validStatusCodes.isEmpty()) {
      StringJoiner validStatusCodesJoiner = new StringJoiner(", ");
      validStatusCodes.forEach(validStatusCodesJoiner::add);
      methodBody.add(".validStatusCodes($L)", validStatusCodesJoiner);
    }

    if (testConnectionConfig.getStatusCodeValidationConfig() != null &&
        testConnectionConfig.getStatusCodeValidationConfig().getErrorTemplateExpression() != null) {
      methodBody.add(".statusCodeValidationErrorTemplateExpression($S)",
                     testConnectionConfig.getStatusCodeValidationConfig().getErrorTemplateExpression());
    }

    for (TestConnectionValidationConfig validationConfig : testConnectionConfig.getValidations()) {
      if (isNotBlank(validationConfig.getValidationExpression())) {
        if (isNotBlank(validationConfig.getErrorTemplateExpression())) {
          methodBody.add(".addValidation($S, $S)", validationConfig.getValidationExpression(),
                         validationConfig.getErrorTemplateExpression());
        } else {
          methodBody.add(".addValidation($S)", validationConfig.getValidationExpression());
        }
      }
    }

    if (testConnectionConfig.getMediaType() != null) {
      methodBody.add(".responseMediaType($T.parse($S))", MediaType.class, testConnectionConfig.getMediaType().toString());
    }

    methodBody.add(".build();");

    methodBody.addStatement("return $L($L, $L)",
                            VALIDATE_CONNECTION_METHOD_NAME,
                            VALIDATE_CONNECTION_CONNECTION_PARAMETER_NAME,
                            VALIDATE_CONNECTION_SETTINGS_VAR_NAME);

    return methodBody.build();
  }
}
