/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.connection.layers;

import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isBlank;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme.SecuritySchemeType.BEARER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.util.SdkTemplatingUtils.getDisplayNameAnnotation;
import static javax.lang.model.element.Modifier.PUBLIC;
import static org.apache.commons.lang.WordUtils.capitalize;

import org.mule.runtime.extension.api.annotation.Alias;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.connection.authentication.SdkAuthenticationStrategy;

import java.nio.file.Path;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public class SdkConnectorProviderImplementationLayer extends AbstractSdkConnectionProviderLayer {

  public static final String IMPLEMENTATION_LAYER_JAVADOC =
      "This is the last layer of the connection provider generation gap pattern. "
          + "It is what finally gets exposed as the connection provider, "
          + "including all customizations made in the refinement layer.";

  public static final String BEARER_PREFIX =
      "bearer-";

  private final TypeName baseClass;

  public SdkConnectorProviderImplementationLayer(Path outputDir,
                                                 ConnectorModel connectorModel,
                                                 ConnectorSecurityScheme securityScheme,
                                                 SdkAuthenticationStrategy authenticationStrategy,
                                                 TypeName baseClass,
                                                 RestSdkRunConfiguration runConfiguration) {
    super(outputDir, connectorModel, securityScheme, authenticationStrategy, runConfiguration);

    this.baseClass = baseClass;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    generateConnectionProviderClass();
  }

  private void generateConnectionProviderClass() throws TemplatingException {
    TypeSpec.Builder connectionProviderClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(PUBLIC)
            .superclass(baseClass)
            .addAnnotation(generateAliasAnnotation())
            .addAnnotation(getDisplayNameAnnotation(getConnectionProviderDisplayName()))
            .addJavadoc(IMPLEMENTATION_LAYER_JAVADOC);

    authenticationStrategy.addClassAnnotations(connectionProviderClassBuilder);

    writeClassToFile(connectionProviderClassBuilder.build(), getPackage());
  }

  private AnnotationSpec generateAliasAnnotation() {
    return AnnotationSpec
        .builder(Alias.class)
        .addMember(VALUE_MEMBER, "$S", getSchemaAlias())
        .build();
  }

  private String getSchemaAlias() {
    if (securityScheme.getSchemeType().equals(BEARER)) {
      return BEARER_PREFIX + (!isBlank(securityScheme.getAlias()) ? securityScheme.getAlias()
          : getConnectionProviderXmlName());
    } else {
      return !isBlank(securityScheme.getAlias()) ? securityScheme.getAlias() : getConnectionProviderXmlName();
    }
  }

  private String getConnectionProviderDisplayName() {
    return capitalize(getConnectionProviderXmlName().replace('-', ' ')) + " Connection Provider";
  }

  public String getJavaClassName() {
    return getJavaUpperCamelNameFromXml(getConnectionProviderXmlName()) + CONNECTION_PROVIDER_CLASSNAME_SUFFIX;
  }

  public String getPackage() {
    return getConnectionProviderBasePackage();
  }
}
