/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.metadata;

import static com.mulesoft.connectivity.rest.sdk.templating.sdk.util.MediaTypeUtil.isJsonCompatibleMediaType;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.util.SdkTemplatingUtils.HAS_PAGINATION_METHOD_NAME;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.util.SdkTemplatingUtils.generateReturnTrueMethod;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.commons.api.datasense.metadata.output.BooleanOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.metadata.output.DateOnlyOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.metadata.output.DateTimeOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.metadata.output.FileOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.metadata.output.FromJsonOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.metadata.output.IntegerOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.metadata.output.JsonOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.metadata.output.NumberOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.metadata.output.StringOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.metadata.output.TimeOnlyOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.metadata.output.XmlOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.metadata.output.dynamic.FromJsonOutputDynamicMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.metadata.output.dynamic.JsonOutputDynamicMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.metadata.output.dynamic.XmlOutputDynamicMetadataResolver;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.PrimitiveTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.JsonTypeSchema;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.TypeSchema;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.XmlTypeSchema;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.util.FileGenerationUtil;

import java.nio.file.Path;

import com.squareup.javapoet.TypeSpec;

public class SdkOutputMetadataResolver extends SdkAbstractStaticMetadataResolver {

  private String qName;
  private final boolean hasPagination;

  public SdkOutputMetadataResolver(Path outputDir,
                                   ConnectorModel connectorModel,
                                   SdkConnector sdkConnector,
                                   String internalName,
                                   TypeDefinition typeDefinition,
                                   String classNameOverride,
                                   boolean hasPagination,
                                   RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, sdkConnector, internalName, typeDefinition, FileGenerationUtil.SchemaNameType.OUTPUT, EMPTY,
          classNameOverride, runConfiguration);
    this.hasPagination = hasPagination;

    if (hasPagination && !(typeDefinition.getTypeSchema() instanceof JsonTypeSchema)) {
      throw new TemplatingException("TypeSchema not supported with Pagination (Only Json Schemas are supported). This is a bug.");
    }
  }

  protected Class<?> buildSuperclass(TypeDefinition typeDefinition) throws TemplatingException {
    TypeSchema typeSchema = typeDefinition.getTypeSchema();

    if (typeDefinition instanceof PrimitiveTypeDefinition) {
      return getPrimitiveMetadataResolverClass((PrimitiveTypeDefinition) typeDefinition);

    } else if (typeSchema instanceof XmlTypeSchema) {
      qName = ((XmlTypeSchema) typeDefinition.getTypeSchema()).getElementName();
      return runConfiguration.useDynamicTypeResolvers() ? XmlOutputDynamicMetadataResolver.class
          : XmlOutputMetadataResolver.class;

    } else if (typeSchema instanceof JsonTypeSchema && isJsonCompatibleMediaType(typeDefinition)) {
      return runConfiguration.useDynamicTypeResolvers() ? JsonOutputDynamicMetadataResolver.class
          : JsonOutputMetadataResolver.class;

    } else if (typeSchema instanceof JsonTypeSchema) {
      return runConfiguration.useDynamicTypeResolvers() ? FromJsonOutputDynamicMetadataResolver.class
          : FromJsonOutputMetadataResolver.class;
    }

    return StringOutputMetadataResolver.class;
  }

  @Override
  protected String buildGetResolverName(String internalName) {
    return internalName + "-output-type-resolver";
  }

  protected Class<?> getPrimitiveMetadataResolverClass(PrimitiveTypeDefinition primitiveTypeDefinition)
      throws TemplatingException {
    PrimitiveTypeDefinition.PrimitiveType primitiveType = primitiveTypeDefinition.getPrimitiveType();

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.NUMBER)) {
      return NumberOutputMetadataResolver.class;
    }

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.INTEGER)) {
      return IntegerOutputMetadataResolver.class;
    }

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.LONG)) {
      return IntegerOutputMetadataResolver.class;
    }

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.STRING)) {
      return StringOutputMetadataResolver.class;
    }

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.BOOLEAN)) {
      return BooleanOutputMetadataResolver.class;
    }

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.DATE_TIME)
        || primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.DATE_TIME_ONLY)
        || primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.ZONED_DATE_TIME)
        || primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.LOCAL_DATE_TIME)) {
      return DateTimeOutputMetadataResolver.class;
    }

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.TIME_ONLY)) {
      return TimeOnlyOutputMetadataResolver.class;
    }

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.DATE_ONLY)
        || primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.DATE)) {
      return DateOnlyOutputMetadataResolver.class;
    }

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.FILE)) {
      return FileOutputMetadataResolver.class;
    }

    throw new TemplatingException(format("Primitive type '%s' not supported. This is a bug.", primitiveType.name()));
  }

  @Override
  protected String getClassNameSuffix() {
    return "OutputMetadataResolver";
  }

  @Override
  protected boolean requiresQNameMethod() {
    return superclass.equals(XmlOutputMetadataResolver.class)
        || superclass.equals(XmlOutputDynamicMetadataResolver.class);
  }

  @Override
  protected boolean requiresFormatMethod() {
    return superclass.equals(FromJsonOutputMetadataResolver.class)
        || superclass.equals(FileOutputMetadataResolver.class)
        || superclass.equals(FromJsonOutputDynamicMetadataResolver.class);
  }

  @Override
  protected String getQName() {
    return qName;
  }

  @Override
  protected boolean requiresToWriteSchema() {
    return superclass.equals(FromJsonOutputMetadataResolver.class)
        || superclass.equals(JsonOutputMetadataResolver.class)
        || superclass.equals(XmlOutputMetadataResolver.class)
        || superclass.equals(FromJsonOutputDynamicMetadataResolver.class)
        || superclass.equals(JsonOutputDynamicMetadataResolver.class)
        || superclass.equals(XmlOutputDynamicMetadataResolver.class);
  }

  @Override
  protected void generateMethods(TypeSpec.Builder typeResolverClassBuilder) {
    super.generateMethods(typeResolverClassBuilder);
    if (hasPagination) {
      generateReturnTrueMethod(typeResolverClassBuilder, HAS_PAGINATION_METHOD_NAME);
    }
  }

}
