/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.model;

import static com.google.common.base.CaseFormat.LOWER_CAMEL;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.util.SdkTemplatingUtils.generateGetter;
import static com.squareup.javapoet.TypeName.BOOLEAN;
import static com.squareup.javapoet.TypeName.INT;
import static javax.lang.model.element.Modifier.PRIVATE;
import static javax.lang.model.element.Modifier.PUBLIC;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;

import org.mule.runtime.api.meta.ExpressionSupport;
import org.mule.runtime.extension.api.annotation.Alias;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.dsl.xml.TypeDsl;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.http.api.client.proxy.ProxyConfig;
import org.mule.sdk.api.annotation.semantics.connectivity.ConfiguresProxy;
import org.mule.sdk.api.annotation.semantics.connectivity.Host;
import org.mule.sdk.api.annotation.semantics.connectivity.Port;
import org.mule.sdk.api.annotation.semantics.security.Password;
import org.mule.sdk.api.annotation.semantics.security.Username;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;

import java.nio.file.Path;
import java.util.Objects;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import javax.lang.model.element.Modifier;

public class SdkHttpProxyConfig extends JavaTemplateEntity {

  public SdkHttpProxyConfig(Path outputDir, ConnectorModel connectorModel, RestSdkRunConfiguration runConfiguration) {
    super(outputDir, connectorModel, runConfiguration);
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    ClassName className = getClassName();
    TypeSpec.Builder typeSpecBuilder =
        TypeSpec
            .classBuilder(className)
            .addModifiers(PUBLIC)
            .addSuperinterface(ProxyConfig.class)
            .addAnnotation(AnnotationSpec
                .builder(Alias.class)
                .addMember(VALUE_MEMBER, "$S", "proxy")
                .build())
            .addAnnotation(AnnotationSpec
                .builder(TypeDsl.class)
                .addMember("allowTopLevelDefinition", "$L", true)
                .build())
            .addAnnotation(ConfiguresProxy.class);

    FieldSpec hostFieldSpec = FieldSpec.builder(String.class, "host", PRIVATE)
        .addAnnotation(Parameter.class)
        .addAnnotation(AnnotationSpec.builder(Expression.class)
            .addMember(VALUE_MEMBER, "$T.$L", ExpressionSupport.class, NOT_SUPPORTED).build())
        .addAnnotation(Host.class)
        .addJavadoc("Host where the proxy requests will be sent.")
        .build();
    typeSpecBuilder.addField(hostFieldSpec);
    typeSpecBuilder.addMethod(generateGetter(hostFieldSpec, LOWER_CAMEL).addAnnotation(Override.class).build());

    FieldSpec portFieldSpec = FieldSpec.builder(INT, "port", PRIVATE)
        .addAnnotation(Parameter.class)
        .addAnnotation(AnnotationSpec.builder(Expression.class)
            .addMember(VALUE_MEMBER, "$T.$L", ExpressionSupport.class, NOT_SUPPORTED).build())
        .addAnnotation(Port.class)
        .addJavadoc("Port where the proxy requests will be sent.")
        .initializer("$T.MAX_VALUE", Integer.class)
        .build();
    typeSpecBuilder.addField(portFieldSpec);
    typeSpecBuilder.addMethod(generateGetter(portFieldSpec, LOWER_CAMEL).addAnnotation(Override.class).build());

    FieldSpec usernameFieldSpec = FieldSpec.builder(String.class, "username", PRIVATE)
        .addAnnotation(Parameter.class)
        .addAnnotation(AnnotationSpec.builder(Expression.class)
            .addMember(VALUE_MEMBER, "$T.$L", ExpressionSupport.class, NOT_SUPPORTED).build())
        .addAnnotation(Username.class)
        .addJavadoc("The username to authenticate against the proxy.")
        .build();
    typeSpecBuilder.addField(usernameFieldSpec);
    typeSpecBuilder.addMethod(generateGetter(usernameFieldSpec, LOWER_CAMEL).addAnnotation(Override.class).build());

    FieldSpec passwordFieldSpec = FieldSpec.builder(String.class, "password", PRIVATE)
        .addAnnotation(Parameter.class)
        .addAnnotation(AnnotationSpec.builder(Expression.class)
            .addMember(VALUE_MEMBER, "$T.$L", ExpressionSupport.class, NOT_SUPPORTED).build())
        .addAnnotation(Password.class)
        .addJavadoc("The password to authenticate against the proxy.")
        .build();
    typeSpecBuilder.addField(passwordFieldSpec);
    typeSpecBuilder.addMethod(generateGetter(passwordFieldSpec, LOWER_CAMEL).addAnnotation(Override.class).build());

    FieldSpec nonProxyHostsFieldSpec = FieldSpec.builder(String.class, "nonProxyHosts", PRIVATE)
        .addAnnotation(Parameter.class)
        .addAnnotation(AnnotationSpec.builder(Expression.class)
            .addMember(VALUE_MEMBER, "$T.$L", ExpressionSupport.class, NOT_SUPPORTED).build())
        .addJavadoc("A list of comma separated hosts against which the proxy should not be used")
        .build();
    typeSpecBuilder.addField(nonProxyHostsFieldSpec);
    typeSpecBuilder.addMethod(generateGetter(nonProxyHostsFieldSpec, LOWER_CAMEL).addAnnotation(Override.class).build());

    typeSpecBuilder.addMethod(MethodSpec
        .methodBuilder("equals")
        .addModifiers(Modifier.PUBLIC)
        .addAnnotation(Override.class)
        .returns(BOOLEAN)
        .addParameter(Object.class, "o")
        .addCode(CodeBlock.builder()
            .beginControlFlow("if (this == o)")
            .addStatement("return true")
            .endControlFlow()
            .beginControlFlow("if (o == null || getClass() != o.getClass())")
            .addStatement("return false")
            .endControlFlow()
            .addStatement("$L that = ($L) o", className.simpleName(), className.simpleName())
            .addStatement("return port == that.port &&\n" +
                "        $T.equals(host, that.host) &&\n" +
                "        $T.equals(username, that.username) &&\n" +
                "        $T.equals(password, that.password) &&\n" +
                "        $T.equals(nonProxyHosts, that.nonProxyHosts)", Objects.class, Objects.class, Objects.class,
                          Objects.class)
            .build())
        .build());

    typeSpecBuilder.addMethod(MethodSpec
        .methodBuilder("hashCode")
        .addModifiers(Modifier.PUBLIC)
        .addAnnotation(Override.class)
        .returns(INT)
        .addCode(CodeBlock.builder()
            .addStatement("return $T.hash(host, port, username, password, nonProxyHosts)", Objects.class)
            .build())
        .build());

    TypeSpec typeSpec = typeSpecBuilder.build();
    writeClassToFile(typeSpec, className.packageName());
  }

  public TypeName getTypeName() {
    return getClassName();
  }

  private ClassName getClassName() {
    return ClassName.get(connectorModel.getBasePackage() + ".api.proxy", "HttpProxyConfig");
  }

}
