/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.operation;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getParameterJavaName;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static javax.lang.model.element.Modifier.PRIVATE;
import static javax.lang.model.element.Modifier.PROTECTED;
import static javax.lang.model.element.Modifier.PUBLIC;

import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.extension.api.runtime.streaming.PagingProvider;
import org.mule.runtime.http.api.HttpConstants;

import com.mulesoft.connectivity.rest.commons.api.binding.ParameterBinding;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;
import com.mulesoft.connectivity.rest.commons.internal.util.StreamUtils;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.Pagination;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.ArrayTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkParameter;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.util.TypeDefinitionUtil;

import java.lang.reflect.Type;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import org.apache.commons.lang3.tuple.Pair;

public abstract class AbstractSdkPaginationOperation extends AbstractSdkOperation implements SdkPaginationStrategy {

  private final Pagination pagination;
  private static final String BASE_REQUEST_BUILDER_MAIN_METHOD_NAME = "RequestBuilderMain";
  private static final String BASE_PAGING_PROVIDER_MAIN_METHOD_NAME = "PagingProviderMain";
  private static final String PARAMETER_REQUEST_FACTORY = "requestFactory";
  protected static final String DEFAULT_PAGING_PARAMETER = "defaultPagingParameter";
  private static final String PARAMETERS_BINDINGS_REF_NAME = "parameterBindingsRef";
  private static final String CUSTOM_PARAMETERS_BINDINGS_REF_NAME = "customParameterBindingsRef";
  private static final String INITIALISE_QUERY_PARAMS = "initMappedQueryParams";
  private static final String INITIALISE_URI_PARAMS = "initMappedUriParams";
  private static final String INITIALISE_HEADER_PARAMS = "initMappedHeaderParams";
  private static final String DEFAULT_QUERY_PARAM_NAME_CONVENTION = "DefaultQueryParam";
  private static final String DEFAULT_URI_PARAM_NAME_CONVENTION = "DefaultUriParam";
  private static final String DEFAULT_HEADER_PARAM_NAME_CONVENTION = "DefaultHeaderParam";
  private static final String ADD_URI_PARAMS_METHOD_NAME = "addUriParams";
  private static final String ADD_QUERY_PARAMS_METHOD_NAME = "addQueryParams";
  private static final String ADD_HEADER_PARAMS_METHOD_NAME = "addHeaderParams";
  private static final String FILTER_PARAMETERS_BOUND_METHOD = "filterParametersAlreadyBound";
  private static final String QUERY_PARAMS_REQUEST_BINDING = "queryParamsRequestBinding";
  private static final String URI_PARAMS_REQUEST_BINDING = "uriParamsRequestBinding";
  private static final String HEADER_PARAMS_REQUEST_BINDING = "headerParamsRequestBinding";

  public AbstractSdkPaginationOperation(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                                        ConnectorOperation operation, RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, sdkConnector, operation, runConfiguration);
    this.pagination = operation.getPagination();
  }

  public Pagination getPagination() {
    return pagination;
  }

  @Override
  protected void generateOperationMethodsFlow(TypeSpec.Builder operationClassBuilder) throws TemplatingException {
    operationClassBuilder.addMethod(generateBaseMainMethod());
    operationClassBuilder.addMethod(generateRequestBuilderMainMethod());
    operationClassBuilder.addMethod(generatePagingProviderMainMethod());
    operationClassBuilder.addMethod(getExternalPagingProvider());
  }

  protected MethodSpec getExternalPagingProvider() throws TemplatingException {
    MethodSpec.Builder methodSpecBuilder =
        MethodSpec.methodBuilder("get" + getJavaUpperCamelNameFromXml(operation.getInternalName()) + "ExternalPagingProvider");

    methodSpecBuilder.addModifiers(PUBLIC);
    methodSpecBuilder.addAnnotation(org.mule.runtime.extension.api.annotation.Ignore.class);

    methodSpecBuilder.addParameters(generateExternalPagingProviderMethodParameters().parametersSpecList().getLeft());

    methodSpecBuilder.addCode(generateExternalPagingProviderMethodBaseMainBody());

    TypeName returnType = generateMethodReturn();
    if (returnType != null) {
      methodSpecBuilder.returns(returnType);
    }

    return methodSpecBuilder.build();
  }

  protected void addResponseBindings(CodeBlock.Builder block) {
    if (operation.getResponseBindings() != null && !operation.getResponseBindings().get().isEmpty()) {
      block.add(".withResponseBinding(getResponseBindings())");
    }
    block.add(";");
  }

  @Override
  public CodeBlock getPagingMethodOperation() throws TemplatingException {
    CodeBlock.Builder block = CodeBlock.builder();
    block.add(
              "return new $T($S, $L, $L, $L, $S, "
                  + " resolveDefaultResponseMediaType(config), $L, $L, overrides.getResponseTimeoutAsMillis())",
              getPagingProviderClass(),
              pagination.getPaginationParameter(),
              DEFAULT_PAGING_PARAMETER,
              PARAMETER_REQUEST_FACTORY,
              EXPRESSION_LANGUAGE,
              getPayloadExpression(),
              PARAMETER_BINDINGS_NAME,
              CUSTOM_PARAMETER_BINDINGS_NAME);
    addResponseBindings(block);
    return block.build();
  }

  protected SdkParameter getPagingParameter() throws TemplatingException {
    return allQueryParameters.stream()
        .filter(x -> x.getExternalName().equalsIgnoreCase(pagination.getPaginationParameter()))
        .findFirst()
        .orElseThrow(() -> new TemplatingException("Could not get paging parameter, this is a bug."));
  }

  @Override
  public TypeName generateMethodReturn() {
    return ParameterizedTypeName.get(ClassName.get(PagingProvider.class),
                                     TypeName.get(RestConnection.class),
                                     ParameterizedTypeName.get(TypedValue.class, String.class));
  }

  @Override
  protected MessageOutputType getMessageOutputType() {
    ParameterizedTypeName parameterizedTypeName = ParameterizedTypeName.get(List.class, String.class);
    TypeName attribute = TypeName.get(Void.class);
    return new MessageOutputType(parameterizedTypeName, attribute);
  }

  @Override
  protected AbstractSdkOperation.ParametersBuilder generateBaseMainMethodParameters() {
    return AbstractSdkOperation
        .builder()
        .configurationParameter()
        .uriParametersMap()
        .queryParametersMap()
        .headerParametersMap()
        .parameterBinding()
        .customParameterBinding()
        .requestBindingParameter()
        .contentParameter(generateContentParameters(false))
        .requestParameter()
        .configurationOverridesParameter()
        .expressionLanguageParameter();
  }

  @Override
  protected Pair<List<ParameterSpec>, CodeBlock.Builder> generateOperationImplementationMethodParameters() {
    return AbstractSdkOperation
        .builder()
        .withAnnotations()
        .configurationParameter()
        .uriParameters(allUriParameters)
        .queryParameters(generateQueryParameters())
        .headerParameters(allHeaders)
        .auxParameters(auxParameters)
        .contentParameter(generateContentParameters(true))
        .requestParameter()
        .configurationOverridesParameter()
        .parametersSpecList();
  }


  @Override
  public CodeBlock generateOperationMethodBaseMainBody() throws TemplatingException {
    CodeBlock.Builder paginationBody = CodeBlock.builder();

    paginationBody.add(toCursorProviderVariables());

    paginationBody.addStatement("$T<$T,$T> $L = connection -> $L",
                                Function.class,
                                RestConnection.class,
                                RestRequestBuilder.class,
                                PARAMETER_REQUEST_FACTORY,
                                generateRequestBuilderMainCall());

    paginationBody.add(generatePagingProviderCall());

    return paginationBody.build();
  }

  private CodeBlock toCursorProviderVariables() {
    CodeBlock.Builder builder = CodeBlock.builder();

    builder.addStatement("$1T<$2T> $3L = new $1T<>($4T.resolveCursorProvider($5L))", Reference.class, Map.class,
                         PARAMETERS_BINDINGS_REF_NAME,
                         StreamUtils.class, PARAMETER_BINDINGS_NAME);

    builder.addStatement("$1T<$2T> $3L = new $1T<>($4T.resolveCursorProvider($5L))", Reference.class, Map.class,
                         CUSTOM_PARAMETERS_BINDINGS_REF_NAME,
                         StreamUtils.class, CUSTOM_PARAMETER_BINDINGS_NAME);

    return builder.build();
  }

  protected abstract boolean externalizePagingParam();

  @Override
  public CodeBlock generateOperationMainCall(TypeName returnType) {
    final CodeBlock.Builder builder = CodeBlock.builder();
    List<String> params = generateOperationMainCallParameters();

    builder.add(generateAuxParameterBindings());

    builder.addStatement("$T<$T, $T> $L = new $T<>()", Map.class, String.class, Object.class, QUERY_PARAMS_MAP, HashMap.class);
    for (SdkParameter parameter : allQueryParameters.stream()
        .filter(sdkParameter -> !isQueryParamDefinedInPagination(sdkParameter.getExternalName())).collect(toList())) {
      builder.addStatement("$L.put(\"" +
          parameter.getExternalName() + "\"," + getParameterValueStatement(parameter) + ")", QUERY_PARAMS_MAP);
    }
    builder.addStatement("$T<$T, $T> $L = new $T<>()", Map.class, String.class, Object.class, URI_PARAMS_MAP, HashMap.class);
    for (SdkParameter parameter : allUriParameters) {
      builder.addStatement("$L.put(\"" +
          parameter.getExternalName() + "\"," + getParameterValueStatement(parameter) + ")", URI_PARAMS_MAP);
    }
    builder.addStatement("$T<$T, $T> $L = new $T<>()", Map.class, String.class, Object.class, HEADER_PARAMS_MAP, HashMap.class);
    for (SdkParameter parameter : allHeaders) {
      builder.addStatement("$L.put(\"" +
          parameter.getExternalName() + "\"," + getParameterValueStatement(parameter) + ")", HEADER_PARAMS_MAP);
    }
    if (getPagingParameterDefinedInPagination().isPresent() && externalizePagingParam()) {
      builder.addStatement("$L = " + getPagingParameterDefinedInPagination().get().getJavaName(), DEFAULT_PAGING_PARAMETER);
    }

    builder.addStatement("$L $L($L)",
                         "return",
                         getBaseMainMethodName(),
                         params.stream().collect(joining(", ")));
    return builder.build();
  }

  private Optional<SdkParameter> getPagingParameterDefinedInPagination() {
    return allQueryParameters.stream().filter(x -> isQueryParamDefinedInPagination(x.getExternalName())).findFirst();
  }

  @Override
  protected List<String> generateOperationMainCallParameters() {
    return AbstractSdkOperation.builder()
        .configurationParameter()
        .uriParametersMap()
        .queryParametersMap()
        .headerParametersMap()
        .parameterBinding()
        .customParameterBinding()
        .requestBindingMethod()
        .contentParameter(generateContentParameters(false))
        .requestParameter()
        .configurationOverridesParameter()
        .expressionLanguageMethod()
        .parametersSpecStringList();
  }

  private List<String> generateExternalPagingProviderMethodMainCallParameters() {
    return AbstractSdkOperation.builder()
        .configurationParameter()
        .uriParametersMap()
        .queryParametersMap()
        .headerParametersMap()
        .parameterBinding()
        .emptyMapParameter()
        .requestBindingParameter()
        .nullParameter()
        .configurationOverridesParameter()
        .expressionLanguageParameter()
        .parametersSpecStringList();
  }

  private AbstractSdkOperation.ParametersBuilder generateExternalPagingProviderMethodParameters() {
    return AbstractSdkOperation
        .builder()
        .configurationParameter()
        .parameterBinding()
        .requestBindingParameter()
        .configurationOverridesParameter()
        .expressionLanguageParameter();
  }

  private CodeBlock generateExternalPagingProviderMethodBaseMainBody() throws TemplatingException {
    CodeBlock.Builder paginationBody = CodeBlock.builder();

    paginationBody.addStatement("$T<$T, $T> $L = $L($L.$L)", Map.class, String.class, Object.class, QUERY_PARAMS_MAP,
                                INITIALISE_QUERY_PARAMS, REQUEST_BINDINGS_NAME, GET_QUERY_PARAMS);
    paginationBody.addStatement("$T<$T, $T> $L = $L($L.$L)", Map.class, String.class, Object.class, URI_PARAMS_MAP,
                                INITIALISE_URI_PARAMS, REQUEST_BINDINGS_NAME, GET_URI_PARAMS);
    paginationBody.addStatement("$T<$T, $T> $L = $L($L.$L)", Map.class, String.class, Object.class, HEADER_PARAMS_MAP,
                                INITIALISE_HEADER_PARAMS, REQUEST_BINDINGS_NAME, GET_HEADER_PARAMS);

    List<String> params = generateExternalPagingProviderMethodMainCallParameters();

    paginationBody.addStatement("$L $L($L)",
                                "return",
                                getBaseMainMethodName(),
                                params.stream().collect(joining(", ")));

    return paginationBody.build();
  }

  @Override
  protected MethodSpec generateOperationBaseMethod() throws TemplatingException {

    MethodSpec.Builder methodBuilder = MethodSpec
        .methodBuilder(getJavaMethodName())
        .addModifiers(PROTECTED);

    Pair<List<ParameterSpec>, CodeBlock.Builder> parameters = AbstractSdkOperation
        .builder()
        .configurationParameter()
        .uriParameters(allUriParameters)
        .queryParameters(generateQueryParameters())
        .headerParameters(allHeaders)
        .auxParameters(auxParameters)
        .contentParameter(generateContentParameters(false))
        .requestParameter()
        .configurationOverridesParameter()
        .parametersSpecList();

    methodBuilder.addParameters(parameters.getLeft());

    TypeName returnType = generateMethodReturn();

    methodBuilder.addStatement("$1T<$2T,$3T> $4L = new $5T<>()", Map.class, String.class, Object.class,
                               CUSTOM_PARAMETER_BINDINGS_NAME,
                               HashMap.class);

    methodBuilder.addCode(generateOperationMainCall(returnType));

    if (returnType != null) {
      methodBuilder.returns(returnType);
    }

    return methodBuilder.build();
  }

  private String getRequestBuilderMainName() {
    return getJavaMethodName() + BASE_REQUEST_BUILDER_MAIN_METHOD_NAME;
  }

  private String getPagingProviderMainName() {
    return getJavaMethodName() + BASE_PAGING_PROVIDER_MAIN_METHOD_NAME;
  }

  private void generateDefaultPagingParameterAttributeWithDefaultValue(List<FieldSpec> defaultValues) {
    Optional<SdkParameter> pagingParameter = getPagingParameterDefinedInPagination();
    if (pagingParameter.isPresent() && externalizePagingParam()) {
      FieldSpec.Builder fieldSpecBuilder =
          FieldSpec.builder(int.class,
                            DEFAULT_PAGING_PARAMETER,
                            PRIVATE);
      if (pagingParameter.get().getDefaultValue() != null && !pagingParameter.get().getDefaultValue().isEmpty()) {
        try {
          fieldSpecBuilder.initializer("$L", Integer.parseInt(pagingParameter.get().getDefaultValue()));
        } catch (Exception e) {
          // No default value set
        }
      }
      defaultValues.add(fieldSpecBuilder.build());
    }
  }

  private void generateHttpParametersAttributesWithDefaultValue(List<Parameter> parameters, String defaultNamingConvention,
                                                                List<FieldSpec> defaultValues) {
    for (Parameter param : parameters) {
      FieldSpec.Builder fieldSpecBuilder =
          FieldSpec.builder(getParameterPrimitiveJavaType(getTypeDefinition(param.getType())),
                            getParameterJavaName(param.getExternalName(),
                                                 param.getTypeDefinition() instanceof ArrayTypeDefinition)
                                + defaultNamingConvention,
                            PRIVATE);
      if (param.getDefaultValue() != null && !param.getDefaultValue().isEmpty()) {
        if (Number.class.isAssignableFrom(getParameterPrimitiveJavaType(getTypeDefinition(param.getType())).getClass())) {
          fieldSpecBuilder.initializer("$L", param.getDefaultValue());
        } else {
          fieldSpecBuilder.initializer("$S", param.getDefaultValue());
        }
      }
      defaultValues.add(fieldSpecBuilder.build());
    }
  }

  private List<FieldSpec> generateAttributesDefaultValue() throws TemplatingException {
    List<FieldSpec> defaultValues = new ArrayList<>();

    generateDefaultPagingParameterAttributeWithDefaultValue(defaultValues);
    generateHttpParametersAttributesWithDefaultValue(operation.getQueryParameters().stream()
        .filter(sdkParameter -> !isQueryParamDefinedInPagination(sdkParameter.getExternalName())).collect(toList()),
                                                     DEFAULT_QUERY_PARAM_NAME_CONVENTION, defaultValues);
    generateHttpParametersAttributesWithDefaultValue(operation.getUriParameters(),
                                                     DEFAULT_URI_PARAM_NAME_CONVENTION, defaultValues);
    generateHttpParametersAttributesWithDefaultValue(operation.getHeaders(),
                                                     DEFAULT_HEADER_PARAM_NAME_CONVENTION, defaultValues);
    return defaultValues;
  }

  private static Type getParameterPrimitiveJavaType(TypeDefinition typeDefinition) {
    if (typeDefinition instanceof ArrayTypeDefinition) {
      // Should not reach here. Array Types should not depend on this method (Only for it's inner type).
      // Array's will throw this exception as they are supported for query/uri/header parameters.
      throw new IllegalArgumentException("Type is not primitive. Should handle arrays in templating. This is a bug.");
    }

    return TypeDefinitionUtil.getJavaType(typeDefinition);

  }

  private MethodSpec generateHttpDefaultMap(String methodName, String map, String nameConvention, List<SdkParameter> parameters,
                                            String parameterBinding)
      throws TemplatingException {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(methodName)
            .addModifiers(PRIVATE)
            .returns(Map.class);

    methodBuilder.addParameters(Collections.singleton(ParameterSpec.builder(ParameterizedTypeName.get(ClassName.get(List.class),
                                                                                                      TypeName.get(
                                                                                                                   ParameterBinding.class)),
                                                                            parameterBinding)
        .build()));

    methodBuilder.addStatement("$T<$T, $T> $L = new $T<>()", Map.class, String.class, Object.class, map, HashMap.class);

    for (SdkParameter param : parameters) {
      methodBuilder.addStatement("$L.put(\"" + param.getExternalName() +
          "\"," + getParameterJavaName(param.getExternalName(), param.isArrayType()) + "$L" + ")", map, nameConvention);
    }
    methodBuilder.addStatement("return $L($L,$L)", FILTER_PARAMETERS_BOUND_METHOD, map, parameterBinding);
    return methodBuilder.build();
  }

  @Override
  protected void generateInstanceMethods(TypeSpec.Builder builder) throws TemplatingException {
    builder.addFields(generateAttributesDefaultValue());
    builder.addMethod(generateHttpDefaultMap(INITIALISE_QUERY_PARAMS, QUERY_PARAMS_MAP, DEFAULT_QUERY_PARAM_NAME_CONVENTION,
                                             allQueryParameters.stream()
                                                 .filter(sdkParameter -> !isQueryParamDefinedInPagination(sdkParameter
                                                     .getExternalName()))
                                                 .collect(toList()),
                                             QUERY_PARAMS_REQUEST_BINDING));
    builder.addMethod(generateHttpDefaultMap(INITIALISE_URI_PARAMS, URI_PARAMS_MAP, DEFAULT_URI_PARAM_NAME_CONVENTION,
                                             allUriParameters, URI_PARAMS_REQUEST_BINDING));
    builder.addMethod(generateHttpDefaultMap(INITIALISE_HEADER_PARAMS, HEADER_PARAMS_MAP, DEFAULT_HEADER_PARAM_NAME_CONVENTION,
                                             allHeaders, HEADER_PARAMS_REQUEST_BINDING));
  }

  private List<ParameterSpec> getHttpParamsParameterSpecs() {
    List<ParameterSpec> httpParams = new ArrayList<>();
    httpParams.add(ParameterSpec.builder(Map.class, URI_PARAMS_MAP).build());
    httpParams.add(ParameterSpec.builder(Map.class, QUERY_PARAMS_MAP).build());
    httpParams.add(ParameterSpec.builder(Map.class, HEADER_PARAMS_MAP).build());
    return httpParams;
  }

  private CodeBlock generateRequestBuilderMainCall() {
    CodeBlock.Builder requestBuilderCall = CodeBlock.builder();

    AbstractSdkOperation.ParametersBuilder parameters = AbstractSdkOperation.builder();
    parameters.requestParameter();
    parameters.configurationOverridesParameter();
    parameters.connectionParameter();
    parameters.configurationParameter();
    parameters.parameterBinding();
    parameters.customParameterBinding();
    parameters.requestBindingParameter();

    List<String> parametersParsed = parameters.parametersSpecList().getLeft().stream()
        .map(x -> x.name)
        .map(name -> {
          if (name.equals(PARAMETER_BINDINGS_NAME)) {
            return PARAMETERS_BINDINGS_REF_NAME + ".get()";
          } else if (name.equals(CUSTOM_PARAMETER_BINDINGS_NAME)) {
            return CUSTOM_PARAMETERS_BINDINGS_REF_NAME + ".get()";
          } else {
            return name;
          }
        })
        .collect(toList());
    parametersParsed.addAll(getHttpParamsParameterSpecs().stream().map(param -> param.name).collect(toList()));
    parametersParsed.addAll(AbstractSdkOperation.builder().expressionLanguageParameter().parametersSpecStringList());

    String parametersSpec = "$1L(" + parametersParsed.stream().collect(joining(", ")) + ")";

    requestBuilderCall.add(parametersSpec, getRequestBuilderMainName());

    return requestBuilderCall.build();
  }

  private CodeBlock generatePagingProviderCall() throws TemplatingException {
    CodeBlock.Builder requestBuilderCall = CodeBlock.builder();

    List<String> parameters = new ArrayList() {

      {
        add(PARAMETER_REQUEST_FACTORY);
        add("config");
        add("overrides");
        add(PARAMETER_BINDINGS_NAME);
        add(CUSTOM_PARAMETER_BINDINGS_NAME);
        add(EXPRESSION_LANGUAGE);
      }
    };
    String parametersSpec = "return $L(" + parameters.stream().collect(joining(", ")) + ")";

    requestBuilderCall.addStatement(parametersSpec, getPagingProviderMainName());

    return requestBuilderCall.build();
  }

  private void generateRequestBuilderMainMethodSignature(MethodSpec.Builder methodSpec) {
    AbstractSdkOperation.ParametersBuilder parameters = AbstractSdkOperation
        .builder();
    parameters.requestParameter();
    parameters.configurationOverridesParameter();
    parameters.connectionParameter();
    parameters.configurationParameter();
    parameters.parameterBinding();
    parameters.customParameterBinding();
    parameters.requestBindingParameter();
    parameters.uriParametersMap();
    parameters.queryParametersMap();
    parameters.headerParametersMap();
    parameters.expressionLanguageParameter();
    methodSpec.addParameters(parameters.parametersSpecList().getLeft());
  }

  public MethodSpec generateRequestBuilderMainMethod() {

    MethodSpec.Builder methodSpecBuilder =
        MethodSpec.methodBuilder(getRequestBuilderMainName())
            .addModifiers(PROTECTED);

    methodSpecBuilder.returns(RestRequestBuilder.class);

    generateRequestBuilderMainMethodSignature(methodSpecBuilder);

    methodSpecBuilder.addCode(generateRestRequestBuilder());

    methodSpecBuilder.addStatement("return builder");

    return methodSpecBuilder.build();
  }

  @Override
  public CodeBlock generateRestRequestBuilder() {
    CodeBlock.Builder builder = CodeBlock.builder();

    builder.add(generateRestRequestBuilderInitialization(getBaseUriString()));

    builder.add(".setQueryParamFormat($L)", QUERY_PARAM_FORMAT_FIELD);

    if (connectorModel.getInterceptors() != null && connectorModel.getInterceptors().size() > 0) {
      builder.add(".$L($L($L))", "responseInterceptorDescriptor", "getResponseInterceptorDescriptor", "config");
    }

    if (operation.getInputMetadata() != null && operation.getInputMetadata().getMediaType() != null) {
      addContentTypeHeader(builder);
    }

    if (operation.getOutputMetadata() != null && operation.getOutputMetadata().getMediaType() != null) {
      builder.add(".$L($S, $S)",
                  ADD_HEADER_METHOD_NAME,
                  ACCEPT_HEADER_NAME,
                  operation.getOutputMetadata().getMediaType().toString());
    }

    if (!allUriParameters.isEmpty()) {
      builder.add(".$L($L)", ADD_URI_PARAMS_METHOD_NAME, URI_PARAMS_MAP);
    }
    if (allQueryParameters.stream().anyMatch(sdkParameter -> !isQueryParamDefinedInPagination(sdkParameter.getExternalName()))) {
      builder.add(".$L($L)", ADD_QUERY_PARAMS_METHOD_NAME, QUERY_PARAMS_MAP);
    }
    if (!allHeaders.isEmpty()) {
      builder.add(".$L($L)", ADD_HEADER_PARAMS_METHOD_NAME, HEADER_PARAMS_MAP);
    }

    addSetBodyMethod(builder);

    builder.add(";");
    return builder.build();
  }

  @Override
  protected CodeBlock generateRestRequestBuilderInitialization(String baseUriString) {
    CodeBlock.Builder builder = CodeBlock.builder();
    AbstractSdkOperation.ParametersBuilder parameters = AbstractSdkOperation.builder();
    parameters.requestParameter();
    parameters.configurationOverridesParameter();
    parameters.connectionParameter();
    parameters.configurationParameter();
    parameters.parameterBinding();
    parameters.customParameterBinding();
    parameters.requestBindingParameter();
    parameters.expressionLanguageParameter();
    builder.add("$T builder = $L($L, $L, $T.$L, " +
        parameters.parametersSpecList().getLeft().stream().map(x -> x.name).collect(joining(", ")) + ")",
                RestRequestBuilder.class,
                GET_REQUEST_BUILDER_WITH_BINDINGS_METHOD,
                baseUriString,
                OPERATION_PATH_FIELD,
                HttpConstants.Method.class,
                operation.getHttpMethod().name().toUpperCase());

    return builder.build();
  }

  private void generatePagingProviderMainMethodSignature(MethodSpec.Builder methodSpec) throws TemplatingException {
    ParameterSpec.Builder parameterBindingsSpec;
    parameterBindingsSpec = ParameterSpec.builder(ParameterizedTypeName.get(ClassName.get(Function.class),
                                                                            TypeName.get(RestConnection.class),
                                                                            TypeName.get(RestRequestBuilder.class)),
                                                  PARAMETER_REQUEST_FACTORY);

    List<ParameterSpec> parameterSpecs = new ArrayList<>();
    parameterSpecs.add(parameterBindingsSpec.build());


    AbstractSdkOperation.ParametersBuilder parameters = AbstractSdkOperation
        .builder()
        .configurationParameter()
        .configurationOverridesParameter()
        .parameterBinding()
        .customParameterBinding()
        .expressionLanguageParameter();

    parameterSpecs.addAll(parameters.parametersSpecList().getLeft());

    methodSpec.addParameters(parameterSpecs);
  }

  public MethodSpec generatePagingProviderMainMethod() throws TemplatingException {

    MethodSpec.Builder methodSpecBuilder =
        MethodSpec.methodBuilder(getPagingProviderMainName())
            .addModifiers(PROTECTED);

    generatePagingProviderMainMethodSignature(methodSpecBuilder);

    methodSpecBuilder.addCode(getPagingMethodOperation());

    TypeName returnType = generateMethodReturn();
    if (returnType != null) {
      methodSpecBuilder.returns(returnType);
    }
    return methodSpecBuilder.build();
  }

  protected List<SdkParameter> generateQueryParametersNoPaginated() {
    if (pagination.getPaginationParameter() != null) {
      return allQueryParameters.stream()
          .filter(x -> !pagination.getPaginationParameter().equals(x.getExternalName()))
          .collect(toList());
    } else {
      return allQueryParameters.stream()
          .collect(toList());
    }
  }

  @Override
  public String getPayloadExpression() {
    return this.pagination.getPaginationResponseExpression();
  }

  @Override
  protected boolean requiresMediaTypeAnnotation() {
    return false;
  }

}
