/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver;

import static javax.lang.model.element.Modifier.ABSTRACT;
import static javax.lang.model.element.Modifier.FINAL;
import static javax.lang.model.element.Modifier.PRIVATE;
import static javax.lang.model.element.Modifier.PROTECTED;
import static javax.lang.model.element.Modifier.STATIC;

import com.mulesoft.connectivity.rest.commons.internal.model.common.DataType;
import com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext;
import com.mulesoft.connectivity.rest.commons.internal.model.common.SimpleEvaluationContext;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverReference;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;

import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

import java.nio.file.Path;
import java.util.Optional;

public abstract class AbstractSdkResolverProviderReferenceParent extends AbstractSdkResolverProvider {

  public static final String GLOBAL_EVALUATION_CONTEXT_FIELD = "globalEvaluationContext";
  public static final String GET_RESOLVER_DECLARATION_METHOD = "getResolverDeclaration";
  public static final String GET_GLOBAL_EVALUATION_CONTEXT_METHOD = "getGlobalEvaluationContext";
  public static final String RESOLVER_NAME_FIELD = "RESOLVER_NAME";

  private final ResolverReference<?> resolverReference;
  private final SdkResolverDefinition<?> sdkResolverDefinition;


  public AbstractSdkResolverProviderReferenceParent(Path outputDir,
                                                    ConnectorModel connectorModel,
                                                    ResolverReference<?> resolverReference,
                                                    SdkResolverDefinition<?> sdkResolverDefinition,
                                                    RestSdkRunConfiguration runConfiguration) {
    super(outputDir, connectorModel, runConfiguration);

    this.resolverReference = resolverReference;
    this.sdkResolverDefinition = sdkResolverDefinition;
  }

  public TypeName getTypeName() {
    return ClassName.get(this.getPackage(), this.getJavaClassName());
  }

  @Override
  protected void buildClass(TypeSpec.Builder classBuilder) {
    classBuilder.addModifiers(ABSTRACT)
        .addField(generateGlobalContextField())
        .addField(getResolverNameField())
        .addMethod(generateGetGlobalEvaluationContext())
        .addMethod(generateGetResolverDeclaration())
        .addModifiers(ABSTRACT);

    sdkResolverDefinition.addClassConstants(classBuilder);
  }

  private MethodSpec generateGetGlobalEvaluationContext() {
    MethodSpec.Builder methodBuilder = MethodSpec
        .methodBuilder(GET_GLOBAL_EVALUATION_CONTEXT_METHOD)
        .addModifiers(PROTECTED)
        .returns(ParameterizedTypeName.get(Optional.class, EvaluationContext.class))
        .addAnnotation(Override.class);

    methodBuilder
        .beginControlFlow("if ($L != null)", GLOBAL_EVALUATION_CONTEXT_FIELD)
        .addStatement("return $T.of($L)", Optional.class, GLOBAL_EVALUATION_CONTEXT_FIELD)
        .endControlFlow()
        .addStatement("$L = new $T()", GLOBAL_EVALUATION_CONTEXT_FIELD, SimpleEvaluationContext.class)
        .addStatement("$L.declare($L())", GLOBAL_EVALUATION_CONTEXT_FIELD, GET_RESOLVER_DECLARATION_METHOD)
        .addStatement("return $T.of($L)", Optional.class, GLOBAL_EVALUATION_CONTEXT_FIELD);

    return methodBuilder.build();
  }

  private MethodSpec generateGetResolverDeclaration() {
    MethodSpec.Builder methodBuilder = MethodSpec
        .methodBuilder(GET_RESOLVER_DECLARATION_METHOD)
        .returns(
                 ParameterizedTypeName
                     .get(com.mulesoft.connectivity.rest.commons.internal.model.resolvers.ResolverDeclaration.class,
                          sdkResolverDefinition.getSdkResolverDefinitionClass()))
        .addModifiers(PRIVATE);

    methodBuilder
        .addStatement("$1T builder = new $1T($2L)",
                      sdkResolverDefinition.getSdkResolverDeclarationBuilderClass(),
                      RESOLVER_NAME_FIELD);

    CodeBlock.Builder builderBuilder = sdkResolverDefinition.getResolverExpressionBuilder();

    for (ResolverParameter resolverParameter : resolverReference.getDeclaration().getParameters()) {
      String dataType = DataType.fromString(resolverParameter.getType().getName()).name().toUpperCase();
      builderBuilder.add(".parameter($S, p -> p.type($T.$L))", resolverParameter.getName(), DataType.class, dataType);
    }

    methodBuilder
        .addStatement(builderBuilder.build())
        .addStatement("return builder.build()");

    return methodBuilder.build();
  }

  private FieldSpec getResolverNameField() {
    return FieldSpec
        .builder(String.class, RESOLVER_NAME_FIELD, PROTECTED, STATIC, FINAL)
        .initializer("$S", resolverReference.getDeclaration().getName())
        .build();
  }

  private FieldSpec generateGlobalContextField() {
    return FieldSpec
        .builder(ClassName.get(EvaluationContext.class), GLOBAL_EVALUATION_CONTEXT_FIELD)
        .addModifiers(PRIVATE, STATIC)
        .build();
  }
}
