/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.trigger.layers;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.operation.AbstractSdkOperation.OPERATION_CLASSNAME_SUFFIX;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.operation.AbstractSdkOperation.OPERATION_PACKAGE;
import static java.lang.String.format;
import static javax.lang.model.element.Modifier.PROTECTED;
import static javax.lang.model.element.Modifier.PUBLIC;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON_TYPE;
import static javax.ws.rs.core.MediaType.APPLICATION_XML_TYPE;
import static javax.ws.rs.core.MediaType.TEXT_PLAIN_TYPE;

import com.mulesoft.connectivity.rest.commons.api.operation.paging.RestPagingProvider;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.HttpConstants;

import com.mulesoft.connectivity.rest.commons.api.binding.HttpRequestBinding;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;
import com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.Trigger;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkParameter;

import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import com.squareup.javapoet.WildcardTypeName;

public class SdkTriggerBaseLayer extends AbstractSdkTriggerLayer {

  private static final String PATH_TEMPLATE_FIELD = "PATH_TEMPLATE";
  private static final String ITEMS_EXPRESSION_FIELD = "ITEMS_EXPRESSION";
  private static final String WATERMARK_EXPRESSION_FIELD = "WATERMARK_EXPRESSION";
  private static final String IDENTITY_EXPRESSION_FIELD = "IDENTITY_EXPRESSION";
  private static final String REQUEST_BODY_EXPRESSION_FIELD = "REQUEST_BODY_EXPRESSION";
  private static final String EVENT_EXPRESSION_FIELD = "EVENT_EXPRESSION";
  private static final String START_VALUE_FIELD = "START_VALUE_EXPRESSION";

  private static final String WATERMARK_DATA_TYPE_METHOD = "getWatermarkDataType";
  private static final String GET_PATH_TEMPLATE_METHOD = "getPathTemplate";
  private static final String GET_REQUEST_BUILDER_METHOD = "getRequestBuilder";
  private static final String GET_PARAMETER_BINDING_METHOD = "getParameterBinding";
  private static final String GET_PARAMETER_VALUES_METHOD = "getParameterValues";
  private static final String GET_CUSTOM_PARAMETER_VALUES_METHOD = "getCustomParameterValues";
  private static final String GET_REQUEST_BODY_DATA_TYPE_METHOD = "getRequestBodyDataType";
  private static final String DO_INSTANCES_START_METHOD = "doInstancesStart";

  private static final String PARAMETER_BINDING_LOCAL_VARIABLE = "parameterBinding";
  private static final String ADD_URI_BINDING_METHOD = "addUriParamBinding";
  private static final String ADD_QUERY_BINDING_METHOD = "addQueryParamBinding";
  private static final String ADD_HEADER_BINDING_METHOD = "addHeaderBinding";

  private static final String PARAMETER_VALUES_LOCAL_VARIABLE = "parameterValues";

  private static final String GET_TYPED_VALUE_OR_NULL_METHOD = "getTypedValueOrNull";

  private final TypeName superclass;
  private final List<SdkParameter> sdkParameters;

  public SdkTriggerBaseLayer(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector, Trigger trigger,
                             List<SdkParameter> sdkParameters, String javaClassName, String packageName, TypeName superclass,
                             RestSdkRunConfiguration runConfiguration) {
    super(outputDir, connectorModel, sdkConnector, trigger, javaClassName, packageName, runConfiguration);

    this.superclass = superclass;

    this.sdkParameters = sdkParameters;
  }

  private MethodSpec generateConstructor() {
    MethodSpec.Builder constructorBuilder =
        MethodSpec.constructorBuilder()
            .addModifiers(PUBLIC);

    if (trigger.getOperation().hasPagination()) {
      constructorBuilder.addStatement("super($L, $L, $L, $L, $L)",
                                      WATERMARK_EXPRESSION_FIELD,
                                      IDENTITY_EXPRESSION_FIELD,
                                      REQUEST_BODY_EXPRESSION_FIELD,
                                      EVENT_EXPRESSION_FIELD,
                                      START_VALUE_FIELD);
    } else {
      constructorBuilder.addStatement("super($L, $L, $L, $L, $L, $L)",
                                      ITEMS_EXPRESSION_FIELD,
                                      WATERMARK_EXPRESSION_FIELD,
                                      IDENTITY_EXPRESSION_FIELD,
                                      REQUEST_BODY_EXPRESSION_FIELD,
                                      EVENT_EXPRESSION_FIELD, START_VALUE_FIELD);
    }


    return constructorBuilder.build();
  }

  private void generateTriggerClass() throws TemplatingException {
    TypeSpec.Builder triggerClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(PUBLIC)
            .superclass(superclass)
            .addMethod(generateConstructor())
            .addMethod(generateWatermarkDataTypeMethod())
            .addMethod(generateGetPathTemplateMethod())
            .addMethod(generateGetParameterBindingMethod())
            .addMethod(generateGetParameterValuesMethod());

    if (trigger.getOperation().hasPagination()) {
      triggerClassBuilder.addMethod(generateDoInstancesStartMethod());
    } else {
      triggerClassBuilder.addMethod(generateGetCustomParameterValuesMethod());
      triggerClassBuilder.addMethod(generateGetRequestBodyDataTypeMethod());
      triggerClassBuilder.addMethod(generateGetRequestBuilderMethod());
    }

    addClassConstants(triggerClassBuilder);
    addParameters(triggerClassBuilder);

    JavaFile.Builder javaFileBuilder = getJavaFileBuilderForClass(triggerClassBuilder.build(), getPackage());
    javaFileBuilder.addStaticImport(RestSdkUtils.class, GET_TYPED_VALUE_OR_NULL_METHOD);
    writeJavaFile(javaFileBuilder.build());
  }

  public String getJavaClassName() {
    return super.getJavaClassName() + BASE_CLASSNAME_SUFFIX;
  }

  public String getPackage() {
    return super.getPackage() + BASE_PACKAGE_SUFFIX;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    generateTriggerClass();
  }

  private MethodSpec generateGetRequestBodyDataTypeMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_REQUEST_BODY_DATA_TYPE_METHOD)
            .addModifiers(PROTECTED)
            .returns(String.class)
            .addAnnotation(Override.class);

    methodBuilder.addStatement("return \"$L\"", getRequestBodyDataType());

    return methodBuilder.build();
  }

  private String getRequestBodyDataType() {
    TypeDefinition inputMetadata = trigger.getOperation().getInputMetadata();

    if (inputMetadata == null || inputMetadata.getMediaType() == null) {
      return MediaType.APPLICATION_JSON.toString();
    }

    if (inputMetadata.getMediaType().isCompatible(APPLICATION_JSON_TYPE)) {
      return MediaType.APPLICATION_JSON.toString();
    } else if (inputMetadata.getMediaType().isCompatible(APPLICATION_XML_TYPE)) {
      return MediaType.APPLICATION_XML.toString();
    } else if (inputMetadata.getMediaType().isCompatible(TEXT_PLAIN_TYPE)) {
      return MediaType.TEXT.toString();
    }

    return MediaType.APPLICATION_JSON.toString();
  }

  private void addParameters(TypeSpec.Builder triggerClassBuilder) {
    for (SdkParameter sdkParameter : sdkParameters) {
      triggerClassBuilder.addField(sdkParameter.generateParameterField().build());
    }
  }

  private void addClassConstants(TypeSpec.Builder triggerClassBuilder) {
    triggerClassBuilder
        .addField(getConstantStringField(PATH_TEMPLATE_FIELD, trigger.getOperation().getPath()))
        .addField(getConstantStringField(ITEMS_EXPRESSION_FIELD, trigger.getItemsExpression()))
        .addField(getConstantStringField(WATERMARK_EXPRESSION_FIELD, trigger.getWatermarkExpression()))
        .addField(getConstantStringField(IDENTITY_EXPRESSION_FIELD, trigger.getIdentityExpression()))
        .addField(getConstantStringField(REQUEST_BODY_EXPRESSION_FIELD, trigger.getRequestBodyExpression()))
        .addField(getConstantStringField(EVENT_EXPRESSION_FIELD, trigger.getEventExpression()))
        .addField(getConstantStringField(START_VALUE_FIELD, trigger.getStartValueExpression()));
  }

  private MethodSpec generateWatermarkDataTypeMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(WATERMARK_DATA_TYPE_METHOD)
            .addModifiers(PROTECTED)
            .returns(DataType.class)
            .addAnnotation(Override.class);

    methodBuilder.addStatement("return $T.fromType($T.class)", DataType.class, getJavaType(trigger.getWatermarkType()));

    return methodBuilder.build();
  }

  private MethodSpec generateDoInstancesStartMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(DO_INSTANCES_START_METHOD)
            .addModifiers(PROTECTED)
            .returns(void.class)
            .addAnnotation(Override.class);

    methodBuilder.addStatement("baseOperationPagingProvider = ($T) new "
        + "$T().$L(config, $L, $L,overrides ,getExpressionLanguage())",
                               RestPagingProvider.class,
                               ClassName.get(connectorModel.getBasePackage() + OPERATION_PACKAGE,
                                             getJavaUpperCamelNameFromXml(trigger.getOperation().getInternalName())
                                                 + OPERATION_CLASSNAME_SUFFIX),
                               "get" + getJavaUpperCamelNameFromXml(trigger.getOperation().getInternalName())
                                   + "ExternalPagingProvider",
                               GET_PARAMETER_VALUES_METHOD + "()", GET_PARAMETER_BINDING_METHOD + "()");
    return methodBuilder.build();
  }

  private MethodSpec generateGetPathTemplateMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_PATH_TEMPLATE_METHOD)
            .addModifiers(PROTECTED)
            .returns(String.class)
            .addAnnotation(Override.class);

    methodBuilder.addStatement("return $L", PATH_TEMPLATE_FIELD);

    return methodBuilder.build();
  }

  private MethodSpec generateGetRequestBuilderMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_REQUEST_BUILDER_METHOD)
            .addModifiers(PROTECTED)
            .returns(RestRequestBuilder.class)
            .addAnnotation(Override.class)
            .addParameter(String.class, "path");

    methodBuilder.addStatement("$1T builder = new $1T(connection.getBaseUri(), path, $2T.$3L)",
                               RestRequestBuilder.class,
                               HttpConstants.Method.class,
                               trigger.getOperation().getHttpMethod().name().toUpperCase());

    if (connectorModel.getInterceptors() != null && connectorModel.getInterceptors().size() > 0) {
      methodBuilder.addStatement("builder.responseInterceptorDescriptor(config.getResponseInterceptorDescriptor())");
    }

    methodBuilder.addStatement("return builder");

    return methodBuilder.build();
  }

  private MethodSpec generateGetParameterBindingMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_PARAMETER_BINDING_METHOD)
            .addModifiers(PROTECTED)
            .returns(HttpRequestBinding.class)
            .addAnnotation(Override.class);

    methodBuilder.addStatement("$1T $2L = new $1T()",
                               HttpRequestBinding.class,
                               PARAMETER_BINDING_LOCAL_VARIABLE);

    if (trigger.getParameterBindings() != null) {
      for (ParameterBinding binding : trigger.getParameterBindings()) {
        methodBuilder.addStatement("$L.$L($S, $S)",
                                   PARAMETER_BINDING_LOCAL_VARIABLE,
                                   getParameterBindingAddMethodName(binding),
                                   binding.getName(),
                                   binding.getExpression());
      }
    }

    methodBuilder.addStatement("return $1L", PARAMETER_BINDING_LOCAL_VARIABLE);

    return methodBuilder.build();
  }

  private String getParameterBindingAddMethodName(ParameterBinding binding) {
    switch (binding.getParameterType()) {
      case QUERY:
        return ADD_QUERY_BINDING_METHOD;
      case HEADER:
        return ADD_HEADER_BINDING_METHOD;
      case URI:
        return ADD_URI_BINDING_METHOD;
    }

    throw new IllegalArgumentException(format("Parameter type '%s' not supported. This is a bug.", binding.getParameterType()));
  }

  private MethodSpec generateGetParameterValuesMethod() {

    if (this.trigger.getOperation().hasPagination()) {
      MethodSpec.Builder methodBuilder =
          MethodSpec.methodBuilder(GET_PARAMETER_VALUES_METHOD)
              .addModifiers(PROTECTED)
              .returns(ParameterizedTypeName.get(ClassName.get(Map.class), ClassName.get(String.class),
                                                 ClassName.get(Object.class)))
              .addAnnotation(Override.class);

      methodBuilder.addStatement("final $T $L = new $T<>()",
                                 ParameterizedTypeName.get(ClassName.get(Map.class), ClassName.get(String.class),
                                                           ClassName.get(Object.class)),
                                 PARAMETER_VALUES_LOCAL_VARIABLE,
                                 HashMap.class);

      for (SdkParameter sdkParameter : sdkParameters) {
        methodBuilder.addStatement("$L.put($S, $L)",
                                   PARAMETER_VALUES_LOCAL_VARIABLE,
                                   sdkParameter.getExternalName(),
                                   sdkParameter.getJavaName());
      }
      methodBuilder.addStatement("return $L", PARAMETER_VALUES_LOCAL_VARIABLE);

      return methodBuilder.build();
    } else {
      MethodSpec.Builder methodBuilder =
          MethodSpec.methodBuilder(GET_PARAMETER_VALUES_METHOD)
              .addModifiers(PROTECTED)
              .returns(getParameterValuesMultiMapType())
              .addAnnotation(Override.class);

      methodBuilder.addStatement("final $T $L = new $T<>()",
                                 getParameterValuesMultiMapType(),
                                 PARAMETER_VALUES_LOCAL_VARIABLE,
                                 MultiMap.class);

      for (SdkParameter sdkParameter : sdkParameters) {
        methodBuilder.addStatement("$L.put($S, $L($L))",
                                   PARAMETER_VALUES_LOCAL_VARIABLE,
                                   sdkParameter.getExternalName(),
                                   GET_TYPED_VALUE_OR_NULL_METHOD,
                                   sdkParameter.getJavaName());
      }

      methodBuilder.addStatement("return $L", PARAMETER_VALUES_LOCAL_VARIABLE);

      return methodBuilder.build();
    }
  }

  private MethodSpec generateGetCustomParameterValuesMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_CUSTOM_PARAMETER_VALUES_METHOD)
            .addModifiers(PROTECTED)
            .returns(getParameterValuesMultiMapType())
            .addAnnotation(Override.class);

    methodBuilder.addStatement("final $T $L = new $T<>()",
                               getParameterValuesMultiMapType(),
                               PARAMETER_VALUES_LOCAL_VARIABLE,
                               MultiMap.class);

    methodBuilder.addComment("Add custom parameters here");

    methodBuilder.addStatement("return $L", PARAMETER_VALUES_LOCAL_VARIABLE);

    return methodBuilder.build();
  }

  private ParameterizedTypeName getParameterValuesMultiMapType() {
    ParameterizedTypeName wildcardTypedValueType =
        ParameterizedTypeName.get(
                                  ClassName.get(TypedValue.class),
                                  WildcardTypeName.subtypeOf(Object.class));

    return ParameterizedTypeName.get(ClassName.get(MultiMap.class), ClassName.get(String.class), wildcardTypedValueType);
  }
}
