/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.valueprovider;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity.getConstantStringField;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver.SdkResolverUtil.getBindingMethod;
import static javax.lang.model.element.Modifier.PROTECTED;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.commons.internal.model.builder.valueprovider.ValueProviderResolverDeclarationBuilder;
import com.mulesoft.connectivity.rest.commons.internal.model.builder.valueprovider.ValueProviderResolverExpressionBuilder;
import com.mulesoft.connectivity.rest.commons.internal.model.valueprovider.ValueProviderResolverDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.httprequest.HttpRequestBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.Argument;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverExpression;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverReference;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.valueprovider.ValueProviderDefinition;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver.SdkResolverDefinition;

import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public class SdkValueProviderDefinition implements SdkResolverDefinition<ValueProviderDefinition> {

  public static final String ITEMS_EXPRESSION_FIELD = "ITEMS_EXPRESSION";
  public static final String ITEM_VALUE_EXPRESSION_FIELD = "ITEM_VALUE_EXPRESSION";
  public static final String ITEM_NAME_EXPRESSION_FIELD = "ITEM_NAME_EXPRESSION";
  public static final String PATH_TEMPLATE_FIELD = "PATH";
  public static final String SCRIPT_TEMPLATE_FIELD = "SCRIPT";

  private final ValueProviderDefinition definition;
  private final boolean isTrigger;

  public SdkValueProviderDefinition(ResolverExpression<ValueProviderDefinition> valueProviderExpression, boolean isTrigger) {
    this.definition = getValueProviderDefinition(valueProviderExpression);
    this.isTrigger = isTrigger;
  }

  private ValueProviderDefinition getValueProviderDefinition(ResolverExpression<ValueProviderDefinition> valueProviderExpression) {

    if (valueProviderExpression instanceof ValueProviderDefinition) {
      return (ValueProviderDefinition) valueProviderExpression;
    } else if (valueProviderExpression instanceof ResolverReference) {
      ResolverReference<ValueProviderDefinition> valueProviderReference =
          (ResolverReference<ValueProviderDefinition>) valueProviderExpression;

      return valueProviderReference.getDeclaration().getResolverDefinition();
    }

    throw new IllegalArgumentException("Invalid valueProviderExpression. This is a bug.");
  }

  @Override
  public CodeBlock.Builder getResolverExpressionBuilder() {
    CodeBlock.Builder codeBlockBuilder = CodeBlock.builder();
    if (definition.getRequest() != null) {
      codeBlockBuilder.add(
                           "builder.definition(definitionBuilder -> definitionBuilder.httpRequest(httpRequestBuilder -> httpRequestBuilder.path($1L).method($2S)",
                           PATH_TEMPLATE_FIELD, definition.getRequest().getMethod());
      if (isNotBlank(definition.getRequest().getOutputMediaType())) {
        codeBlockBuilder.add(".outputMediaType($1S)", definition.getRequest().getOutputMediaType());
      }

      HttpRequestBinding bindings = definition.getRequest().getHttpRequestBinding();
      if (bindings != null) {
        codeBlockBuilder.add(".bindings(bindingBuilder -> bindingBuilder");
        bindings.getHeader().forEach(binding -> codeBlockBuilder.add(getBindingCode(binding, HEADER)));
        bindings.getQueryParameter().forEach(binding -> codeBlockBuilder.add(getBindingCode(binding, QUERY)));
        bindings.getUriParameter().forEach(binding -> codeBlockBuilder.add(getBindingCode(binding, URI)));
        codeBlockBuilder.add(")");
      }
    } else {
      codeBlockBuilder.add(
                           "builder.definition(definitionBuilder -> definitionBuilder.script(script -> script.scriptExpression(dw -> dw.expression($1L))",
                           SCRIPT_TEMPLATE_FIELD);
    }

    codeBlockBuilder.add(")");

    codeBlockBuilder.add(".itemExtractionExpression($1L).itemValueExpression($2L).itemNameExpression($3L))",
                         ITEMS_EXPRESSION_FIELD,
                         ITEM_VALUE_EXPRESSION_FIELD,
                         ITEM_NAME_EXPRESSION_FIELD);

    return codeBlockBuilder;
  }

  private static CodeBlock getBindingCode(Argument binding, ParameterType parameterType) {
    CodeBlock.Builder bindingCodeBuilder = CodeBlock.builder();

    bindingCodeBuilder
        .add(".$1L($2S, argumentBuilder -> argumentBuilder.value(expressionBuilder -> expressionBuilder.expression($3S)))",
             getBindingMethod(parameterType),
             binding.getName(),
             binding.getValue().getValue());

    return bindingCodeBuilder.build();
  }

  @Override
  public void addClassConstants(TypeSpec.Builder valueProviderClassBuilder) {
    valueProviderClassBuilder
        .addField(getConstantStringField(ITEMS_EXPRESSION_FIELD, definition.getItemExtractionExpression()))
        .addField(getConstantStringField(ITEM_VALUE_EXPRESSION_FIELD, definition.getItemValueExpression()))
        .addField(getConstantStringField(ITEM_NAME_EXPRESSION_FIELD, definition.getItemDisplayNameExpression()));
    if (definition.getRequest() != null) {
      valueProviderClassBuilder.addField(getConstantStringField(PATH_TEMPLATE_FIELD, definition.getRequest().getPath()));
    } else {
      valueProviderClassBuilder.addField(getConstantStringField(SCRIPT_TEMPLATE_FIELD, definition.getScript()));
    }
  }

  @Override
  public void addBuildMethod(TypeSpec.Builder classBuilder, CodeBlock methodBody) {
    MethodSpec.Builder buildMethodBuilder =
        MethodSpec.methodBuilder("build")
            .addModifiers(PROTECTED)
            .addParameter(TypeName.get(ValueProviderResolverExpressionBuilder.class), "builder")
            .addAnnotation(Override.class);

    buildMethodBuilder.addStatement(methodBody);

    classBuilder.addMethod(buildMethodBuilder.build());
  }

  @Override
  public ValueProviderDefinition getDefinition() {
    return definition;
  }

  @Override
  public Class<?> getSdkResolverDefinitionClass() {
    return ValueProviderResolverDefinition.class;
  }

  @Override
  public Class<?> getSdkResolverDeclarationBuilderClass() {
    return ValueProviderResolverDeclarationBuilder.class;
  }

  @Override
  public String getEvaluationContextKind() {
    return isTrigger ? EVALUATION_CONTEXT_KIND_TRIGGER : EVALUATION_CONTEXT_KIND_OPERATION;
  }
}
