/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.util;

import static java.util.Arrays.stream;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.StringUtils.stripAccents;
import static org.apache.commons.lang3.text.translate.EntityArrays.BASIC_ESCAPE;

import javax.validation.constraints.NotNull;

public class XmlUtils {

  private XmlUtils() {}

  public static boolean isXmlSchema(String schema) {
    return isNotBlank(schema) && schema.trim().startsWith("<");
  }

  /**
   * @param artifact The artifact string to be sanitized.
   * @return Returns a sanitized string that is a valid Maven POM artifact.
   */
  public static String removeMavenArtifactUnwantedCharacters(String artifact) {
    if (artifact == null) {
      return null;
    }

    return artifact.replaceAll("[^a-zA-Z0-9_\\-]", EMPTY);
  }

  /**
   * @param group The group string to be sanitized.
   * @return Returns a sanitized string that is a valid Maven POM group.
   */
  public static String removeMavenGroupUnwantedCharacters(String group) {
    if (group == null) {
      return null;
    }

    return group.replaceAll("[^a-zA-Z0-9\\._\\-]", EMPTY);
  }

  /**
   * @param version The version string to be sanitized.
   * @return Returns a sanitized string that is a valid Maven POM version.
   */
  public static String removeMavenVersionUnwantedCharacters(String version) {
    if (version == null) {
      return null;
    }

    return version.replaceAll("[^a-zA-Z0-9\\._\\-]", EMPTY);
  }

  private static String splitCapsWithHypens(String name) {
    return ParserUtils.splitCaps(name, "-");
  }

  private static String removeXmlNameUnwantedCharacters(String name) {
    // TODO RSDK-621 remove utilities naming in favour of leaving the control to the user
    if (name == null) {
      return null;
    }

    // Remove accents from characters
    name = stripAccents(name);

    // Only keep letters, numbers, underscores, hyphens and supported escaped characters (From EntityArrays.BASIC_ESCAPE()).
    StringBuilder sanitizedBuilder = new StringBuilder();
    for (Character c : name.toCharArray()) {
      if ((c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') || (c >= '0' && c <= '9') || c == '_' || c == '-') {
        sanitizedBuilder.append(c);
      } else {
        String[] sanitization = stream(BASIC_ESCAPE()).filter(x -> x[0].equals(c.toString())).findFirst().orElse(null);
        if (sanitization != null) {
          if (sanitizedBuilder.length() > 0 && sanitizedBuilder.charAt(sanitizedBuilder.length() - 1) != '-') {
            sanitizedBuilder.append("-");
          }
          // EntityArrays provides escapes that look like &lt; so we just take the keyword we are interested in (lt).
          sanitizedBuilder.append(sanitization[1], 1, sanitization[1].length() - 1);
          sanitizedBuilder.append("-");
        }
      }
    }

    return sanitizedBuilder.toString();
  }

  /**
   * @param name The string to be sanitized. Not Null.
   * @return Return a valid xml element name. It's format will be an underscore split lowercase string.
   */
  public static String getXmlName(@NotNull String name) {
    name = splitCapsWithHypens(name).toLowerCase();

    while (name.matches(".*[a-zA-Z0-9]+[\\s]+[a-zA-Z0-9]+.*")) {
      name = name.replaceAll("([a-zA-Z0-9])[\\s]+([a-zA-Z0-9])", "$1-$2");
    }

    name = removeXmlNameUnwantedCharacters(name);

    name = name.replaceAll("_+", "-");

    if (isEmpty(name)) {
      return EMPTY;
    }

    if (name.toLowerCase().startsWith("xml") || Character.isDigit(name.charAt(0))) {
      name = "_" + name;
    }

    if (name.startsWith("-")) {
      name = name.substring(1);
    }

    if (name.endsWith("-")) {
      name = name.substring(0, name.length() - 1);
    }

    return name;
  }
}
