/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.runtime;

import org.mule.munit.plugin.maven.AbstractMunitMojo;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Optional.ofNullable;
import static org.apache.commons.lang3.StringUtils.split;

import java.util.List;
import java.util.Optional;
import java.util.Properties;

/**
 * Model of the {@link AbstractMunitMojo} runtime configuration parameter. It holds information about the runtimes which to run
 * the tests against
 *
 * @author Mulesoft Inc.
 * @since 2.3.0
 */
public class RuntimeConfiguration {

  public static final String ADDITIONAL_RUNTIMES_PROPERTY = "additionalRuntimes";
  public static final String ADDITIONAL_RUNTIME_SEPARATOR = ",";

  private DiscoverRuntimes discoverRuntimes;
  private List<String> additionalRuntimes = emptyList();

  public RuntimeConfiguration() {}

  private RuntimeConfiguration(DiscoverRuntimes discoverRuntimes, List<String> additionalRuntimes) {
    this.discoverRuntimes = discoverRuntimes;
    this.additionalRuntimes = additionalRuntimes;
  }

  public Optional<DiscoverRuntimes> getDiscoverRuntimes() {
    return ofNullable(discoverRuntimes);
  }

  public List<String> getAdditionalRuntimes() {
    return additionalRuntimes;
  }

  public static Builder builder() {
    return new Builder();
  }

  /**
   * Builder for a {@link RuntimeConfiguration}
   *
   * @author Mulesoft Inc.
   * @since 2.3.0
   */
  public static class Builder {

    private DiscoverRuntimes discoverRuntimes;
    private List<String> additionalRuntimes = emptyList();

    public Builder from(RuntimeConfiguration runtimeConfiguration) {
      if (runtimeConfiguration == null) {
        return this;
      }
      runtimeConfiguration.getDiscoverRuntimes()
          .ifPresent(discRunt -> this.discoverRuntimes = DiscoverRuntimes.builder().from(discRunt).build());
      this.additionalRuntimes = runtimeConfiguration.getAdditionalRuntimes();
      return this;
    }

    public Builder withDiscoverRuntimes(DiscoverRuntimes discoverRuntimes) {
      this.discoverRuntimes = discoverRuntimes;
      return this;
    }

    public Builder withAdditionalRuntimes(List<String> additionalRuntimes) {
      this.additionalRuntimes = additionalRuntimes;
      return this;
    }

    public Builder withOverrides(Properties userProperties) {
      this.discoverRuntimes = DiscoverRuntimes.builder().from(discoverRuntimes).withOverrides(userProperties).build();
      if (userProperties.containsKey(ADDITIONAL_RUNTIMES_PROPERTY)) {
        this.additionalRuntimes = asList(split(userProperties.getProperty(ADDITIONAL_RUNTIMES_PROPERTY),
                                               ADDITIONAL_RUNTIME_SEPARATOR));
      }
      return this;
    }

    public RuntimeConfiguration build() {
      return new RuntimeConfiguration(discoverRuntimes, additionalRuntimes);
    }
  }
}
