/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.util;

import static java.util.Arrays.asList;
import static org.mule.munit.remote.classloading.ClassLoaderUtils.STARTER_JAR_FILE_NAME;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.logging.Log;

/**
 * <p>
 * Generates the final argument lines for the new JVM
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class ArgLinesManager {

  public static final String JAVA_SYSTEM_CLASS_LOADER = "java.system.class.loader";

  private static final List<String> INVALID_ARG_LINES =
      asList("-jar", "-cp", "-classpath", "-D" + STARTER_JAR_FILE_NAME, "-D" + JAVA_SYSTEM_CLASS_LOADER);

  public static final String LOG4J_CONFIGURATION_FILE_PROPERTY = "log4j.configurationFile";

  private static final String DEFAULT_DEBUG_ARG_LINE = "-agentlib:jdwp=transport=dt_socket,server=y,address=5005,suspend=y";

  private Log log;
  private String starterJarFileName;
  private String debuggerArgLine;
  private File log4jConfiguration;
  private List<String> originalArgLines;

  public ArgLinesManager(List<String> originalArgLines, String starterJarFileName, String debuggerArgLine,
                         File log4jConfiguration, Log log) {
    this.originalArgLines = originalArgLines;
    this.starterJarFileName = starterJarFileName;
    this.log4jConfiguration = log4jConfiguration;
    this.debuggerArgLine = debuggerArgLine;
    this.log = log;
  }

  public List<String> getEffectiveArgLines() {
    List<String> effectiveArgLines = new ArrayList<>();
    for (String argLine : originalArgLines) {
      if (filterArgLine(argLine)) {
        log.warn(argLine + " can not be set. We will ignore this argLine");
      } else {
        effectiveArgLines.add(argLine);
      }
    }
    addDefaultArgLines(effectiveArgLines);
    addDebuggerArgLine(effectiveArgLines);
    addLog4jConfigurationFile(effectiveArgLines);

    return effectiveArgLines;
  }

  private boolean filterArgLine(String argLine) {
    if (StringUtils.isBlank(argLine)) {
      return true;
    }

    for (String invalidArgLine : INVALID_ARG_LINES) {
      if (argLine.startsWith(invalidArgLine)) {
        return true;
      }
    }
    return false;
  }

  private void addDefaultArgLines(List<String> effectiveArgLines) {
    effectiveArgLines.add("-D" + STARTER_JAR_FILE_NAME + "=" + starterJarFileName);
  }

  private void addDebuggerArgLine(List<String> effectiveArgLines) {
    if ("true".equalsIgnoreCase(debuggerArgLine)) {
      effectiveArgLines.add(DEFAULT_DEBUG_ARG_LINE);
    } else if (!"false".equalsIgnoreCase(debuggerArgLine)) {
      effectiveArgLines.add(debuggerArgLine);
    }
  }

  private void addLog4jConfigurationFile(List<String> effectiveArgLines) {
    if (log4jConfiguration.exists()) {
      effectiveArgLines.add("-D" + LOG4J_CONFIGURATION_FILE_PROPERTY + "=" + log4jConfiguration.getAbsolutePath());
    }
  }

}

