/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.tool.verify;

import org.mule.munit.common.behavior.ProcessorCall;
import org.mule.munit.common.behavior.ProcessorId;
import org.mule.munit.mock.tool.AbstractMockingTool;
import org.mule.runtime.api.artifact.Registry;

import java.util.List;
import java.util.Map;

import static org.mule.munit.common.exception.MunitFail.fail;

/**
 * The goal of this class is to validate the execution of a processor
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class Verifier extends AbstractMockingTool {

  public Verifier(Registry registry) {
    super(registry);
  }

  /**
   * Defines the name of the processor to verify call
   *
   * @param name The name of the processor to verify call
   * @return Itself
   */
  public Verifier verifyCallOfProcessor(String name) {
    this.processorName = name;
    return this;
  }


  /**
   * Defines the namespace of the processor to verify call
   *
   * @param namespace The namespace of the processor to verify call
   * @return Itself
   */
  public Verifier ofNamespace(String namespace) {
    this.processorNamespace = namespace;
    return this;
  }

  /**
   * Defines the attributes of the processor to verify call
   *
   * @param attributes a map of attributes name,value
   * @return Itself
   */
  public Verifier withAttributes(Map<String, Object> attributes) {
    this.processorAttributes = attributes;
    return this;
  }

  /**
   * The times it must be called
   *
   * @param times The times it must be called
   */
  public void times(Integer times) {
    List<ProcessorCall> executedCalls = getExecutedCalls();

    if (executedCalls.size() != times) {
      fail("On " + getFullName() + ". Expected " + times + " but got " + executedCalls.size() + " calls");
    }
  }

  /**
   * At least the times it must be called
   *
   * @param atLeast At least the times it must be called
   */
  public void atLeast(Integer atLeast) {
    checkValidQuery();

    List<ProcessorCall> executedCalls = getExecutedCalls();
    if (executedCalls.size() < atLeast) {
      fail("On " + getFullName() + ". Expected at least " + atLeast + " but got " + executedCalls.size() + " calls");
    }
  }

  /**
   * At most the times it must be called
   *
   * @param atMost At most the times it must be called
   */
  public void atMost(Integer atMost) {
    checkValidQuery();

    List<ProcessorCall> executedCalls = getExecutedCalls();
    if (executedCalls.size() > atMost) {
      fail("On " + getFullName() + ". Expected at most " + atMost + " but got " + executedCalls.size() + " calls");
    }
  }

  /**
   * The times it must be called must be in the range of [atLeast, atMost]
   *
   * @param atLeast At least the times it must be called
   * @param atMost At most the times it must be called
   */
  public void between(Integer atLeast, Integer atMost) {
    checkValidQuery();

    Integer numberOfExecutedCalls = getExecutedCalls().size();
    if (numberOfExecutedCalls < atLeast || numberOfExecutedCalls > atMost) {
      String failureMessage = String.format("On %s. Expected to be called between %s and %s times but got %s calls",
                                            getFullName(), atLeast, atMost, numberOfExecutedCalls);
      fail(failureMessage);
    }
  }

  /**
   * At least one time called
   */
  public void atLeastOnce() {
    checkValidQuery();
    List<ProcessorCall> executedCalls = getExecutedCalls();
    if (executedCalls.isEmpty()) {
      fail("On " + getFullName() + ".It was never called");
    }
  }

  private List<ProcessorCall> getExecutedCalls() {
    return getManager().findCallsFor(new ProcessorId(processorName, processorNamespace), processorAttributes);
  }
}
