/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.coverage.model;

import static org.mule.munit.common.util.Preconditions.checkArgument;
import static org.mule.munit.common.util.Preconditions.checkNotNull;
import static java.lang.Double.valueOf;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import java.util.ArrayList;
import java.util.List;

/**
 * <p>
 * A model that represents a Mule Config File
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class MuleResource {

  private String name;
  private Double weight = null;
  private List<MuleFlow> flows = new ArrayList<>();

  private Double branchCoverage = null;

  public MuleResource(String name) {
    checkArgument(isNotBlank(name), "The name must not be null nor empty.");
    this.name = name;
  }

  public String getName() {
    return name;
  }

  public void setWeight(Double weight) {
    this.weight = weight;
  }

  public Double getWeight() {
    return weight;
  }

  public List<MuleFlow> getFlows() {
    return flows;
  }

  public void setFlows(List<MuleFlow> flows) {
    checkNotNull(flows, "The flows list must not be null.");
    this.flows = new ArrayList<>(flows);
  }

  public Double getBranchCoverage() {
    return branchCoverage;
  }

  public String getWeightFormatted() {
    return String.format("%.2f", weight);
  }

  public Double getCoverage() {
    int resourcePathCount = 0;
    int resourceCoveredPathCount = 0;

    for (MuleFlow flow : flows) {
      resourcePathCount += flow.getLocations().size();
      resourceCoveredPathCount += flow.getCoveredLocations().size();
    }

    if (resourcePathCount > 0) {
      return (valueOf(resourceCoveredPathCount) * 100) / valueOf(resourcePathCount);
    } else {
      return -1.00;
    }
  }

  public Integer getMessageProcessorCount() {
    Integer count = 0;
    for (MuleFlow mf : flows) {
      count += mf.getLocations().size();
    }
    return count;
  }

  public Integer getCoveredMessageProcessorCount() {
    Integer count = 0;
    for (MuleFlow mf : flows) {
      count += mf.getCoveredLocations().size();
    }
    return count;
  }


}
