/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.coverage.server;

import org.mule.munit.common.util.Preconditions;

import java.util.Optional;
import java.util.concurrent.TimeUnit;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.util.concurrent.Executors;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;


/**
 * It listens for notifications of the coverage module reporting coverage data
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class CoverageServer {

  private static final int SOCKET_TIMEOUT_MILLIS = 30 * 60 * 1000;

  private transient Logger log = LogManager.getLogger(this.getClass());


  private Integer port;
  private Integer serverTimeout;
  private Boolean running = false;
  private Boolean keepRunning = true;
  private Latch startLatch = new Latch();
  private Thread serverThread;
  private ServerSocket providerSocket = null;

  private CoverageLocationsAccumulator coverageLocationsAccumulator;

  public CoverageServer(Integer port, CoverageLocationsAccumulator coverageLocationsAccumulator) {
    this(port, null, coverageLocationsAccumulator);
  }

  public CoverageServer(Integer port, Integer serverTimeout, CoverageLocationsAccumulator coverageLocationsAccumulator) {
    Preconditions.checkNotNull(port, "The port can not be null.");
    Preconditions.checkArgument(port > 0, "The port must be a positive number.");
    Preconditions.checkNotNull(coverageLocationsAccumulator, "The report accumulator must not be null.");

    this.port = port;
    this.serverTimeout = Optional.ofNullable(serverTimeout).orElse(SOCKET_TIMEOUT_MILLIS);
    this.coverageLocationsAccumulator = coverageLocationsAccumulator;
  }

  public int getPort() {
    return this.port;
  }

  public boolean isRunning() {
    return running;
  }

  public void launch() {
    if (running) {
      throw new RuntimeException("The Coverage server is already running it can not be started again.");
    }

    serverThread = new ServerThread();
    Executors.newSingleThreadExecutor().execute(serverThread);
    try {
      startLatch.await(serverTimeout, TimeUnit.MILLISECONDS);

    } catch (InterruptedException cause) {
      throw new RuntimeException("Coverage server was interrupted when starting", cause);
    }
  }

  public void shutDown() {
    doShutdown();
    if (serverThread != null && !serverThread.isInterrupted()) {
      serverThread.interrupt();
    }
  }

  private class ServerThread extends Thread {

    @Override
    public void run() {
      log.info("Coverage Server starting...");
      running = true;
      keepRunning = true;

      try {
        providerSocket = new ServerSocket(port, 10);
        providerSocket.setSoTimeout(serverTimeout);
        log.info("Waiting for coverage client connection in port [" + port + "]...");
        startLatch.countDown();
        do {
          Socket connection = providerSocket.accept();
          log.info("Coverage client connection received from " + connection.getInetAddress().getHostName() + " - "
              + keepRunning);

          CoverageMessageParser parser =
              new CoverageMessageParser(new ObjectInputStream(connection.getInputStream()), coverageLocationsAccumulator);
          Executors.newSingleThreadExecutor().execute(parser);
        } while (keepRunning);

      } catch (SocketTimeoutException timeoutException) {
        log.warn("Coverage Server time out");
        if (keepRunning) {
          log.error("Coverage Server connection timeout after " + String.valueOf(serverTimeout) + " milliseconds");
        }

      } catch (IOException ioException) {
        if (keepRunning) { // Socket.accept() will fail when stopping regularly. Only log when that is not the case
          log.error("Failed to start Coverage Server in port " + port, ioException);
        }

      } finally {
        doShutdown();
      }
    }
  }

  private void doShutdown() {
    try {
      log.debug("Shutting down coverage server running in port [" + port + "]...");
      running = false;
      keepRunning = false;
      if (null != providerSocket) {
        providerSocket.close();
      }

      log.debug("Coverage server shutdown");

    } catch (IOException ioException) {
      log.debug("Coverage Server error during shut down.", ioException);
    }
  }
}
