/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.processors;

import static org.apache.commons.lang3.Validate.notNull;

import org.mule.munit.common.api.event.EventBuilder;
import org.mule.munit.common.api.model.EventAttributes;
import org.mule.munit.common.api.model.EventError;
import org.mule.munit.common.api.model.NullObject;
import org.mule.munit.common.api.model.Payload;
import org.mule.munit.common.api.model.UntypedEventError;
import org.mule.munit.common.api.model.Variable;
import org.mule.munit.common.exception.MunitError;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.DataTypeBuilder;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.core.privileged.PrivilegedMuleContext;

import java.util.Collections;
import java.util.List;

import javax.inject.Inject;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>
 * Sets the payload
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class SetEventProcessor extends MunitProcessor {

  private static final String MEDIA_TYPE_FIELD = "Media Type";
  private static final String ENCODING_FIELD = "Encoding";
  private static final String KEY_FIELD = "Key";

  private transient Logger logger = LoggerFactory.getLogger(this.getClass());

  private Boolean cloneOriginalEvent = false;
  private final org.mule.munit.common.api.model.Event event = new org.mule.munit.common.api.model.Event();


  @Inject
  PrivilegedMuleContext muleContext;

  @Override
  protected String getProcessor() {
    return "set";
  }

  @Override
  protected CoreEvent doProcess(CoreEvent incomingEvent) {

    EventBuilder builder = cloneOriginalEvent ? new EventBuilder(incomingEvent) : new EventBuilder(incomingEvent.getContext());

    MediaType payloadMediaType = evaluatePayload(event.getPayload(), incomingEvent, builder);

    evaluateMediaType(event.getPayload(), incomingEvent, payloadMediaType, builder);

    evaluateAttributes(event.getAttributes(), incomingEvent, builder);

    evaluateErrorType(event.getError(), incomingEvent, builder);

    evaluateVariables(event.getVariables(), incomingEvent, builder);

    return (CoreEvent) builder.build();
  }

  private void evaluateVariables(List<Variable> variables, CoreEvent incomingEvent, EventBuilder builder) {
    if (null != variables) {
      if (variables.isEmpty()) {
        builder.withVariables(Collections.emptyMap());
      } else {
        variables.forEach(v -> {
          TypedValue variableValue = evaluate(incomingEvent, v.getValue());
          builder.addVariable(evaluateAsString(incomingEvent, v.getKey(), KEY_FIELD),
                              variableValue.getValue(),
                              evaluateMediaTypeExpression(incomingEvent, v.getMediaType(),
                                                          variableValue.getDataType().getMediaType()),
                              evaluateAsString(incomingEvent, v.getEncoding(), ENCODING_FIELD));
        });
      }
    }
  }

  private MediaType evaluatePayload(Payload payload, CoreEvent incomingEvent, EventBuilder builder) {
    if (!(payload.getValue() instanceof NullObject)) {
      TypedValue payloadValue = evaluate(incomingEvent, payload.getValue());
      builder.withPayload(payloadValue.getValue());
      return payloadValue.getDataType().getMediaType();
    }
    return MediaType.ANY;
  }

  private void evaluateMediaType(Payload payload, CoreEvent incomingEvent, MediaType valueMediaType, EventBuilder builder) {
    DataTypeBuilder dataTypeBuilder = DataType.builder();
    MediaType mediaType = null;
    if (cloneOriginalEvent && StringUtils.isNotBlank(payload.getMediaType())) {
      dataTypeBuilder.mediaType(evaluateMediaTypeExpression(incomingEvent, payload.getMediaType(), valueMediaType));
    }

    if (!cloneOriginalEvent) {
      dataTypeBuilder.mediaType(evaluateMediaTypeExpression(incomingEvent, payload.getMediaType(), valueMediaType));
    }

    if (StringUtils.isNotBlank(payload.getEncoding())) {
      dataTypeBuilder.charset(evaluateAsString(incomingEvent, payload.getEncoding(), ENCODING_FIELD));
    }

    if ((cloneOriginalEvent && StringUtils.isNotBlank(payload.getMediaType())) || !cloneOriginalEvent ||
        StringUtils.isNotBlank(payload.getEncoding())) {
      mediaType = dataTypeBuilder.build().getMediaType();
    }
    if (mediaType != null) {
      builder.withMediaType(mediaType);
    }
  }

  private void evaluateAttributes(EventAttributes attributes, CoreEvent incomingEvent, EventBuilder builder) {
    try {
      if (attributes.getValue() != null && !(attributes.getValue() instanceof NullObject)) {
        TypedValue attributesValue = evaluate(incomingEvent, attributes.getValue());
        builder.withAttributes(attributesValue.getValue(),
                               evaluateMediaTypeExpression(incomingEvent, attributes.getMediaType(),
                                                           attributesValue.getDataType().getMediaType()),
                               evaluateAsString(incomingEvent, attributes.getEncoding(), ENCODING_FIELD));
      }

    } catch (ClassCastException e) {
      throw new MunitError("Attributes evaluation failed", e);
    }
  }

  private void evaluateErrorType(EventError error, CoreEvent incomingEvent, EventBuilder builder) {
    ErrorType errorType = null;
    String errorTypeId = evaluateAsString(incomingEvent, error.getTypeId(), "Error Type Id");
    if (StringUtils.isNotBlank(errorTypeId) && null != evaluate(incomingEvent, error.getCause())) {
      ComponentIdentifier componentIdentifier = ComponentIdentifier.buildFromStringRepresentation(errorTypeId);
      errorType = muleContext.getErrorTypeLocator().lookupComponentErrorType(componentIdentifier,
                                                                             evaluateErrorCause(error, incomingEvent));
    }
    if (null != errorType) {
      builder.withError(errorType, evaluateErrorCause(error, incomingEvent));
    }
  }

  private Throwable evaluateErrorCause(EventError error, CoreEvent incomingEvent) {
    try {
      return (Throwable) evaluate(incomingEvent, error.getCause()).getValue();
    } catch (ClassCastException e) {
      throw new MunitError(String.format("Error cause '%s' should be Throwable", error.getCause()), e);
    }
  }

  private String evaluateAsString(CoreEvent event, Object expression, String name) {
    return expressionWrapper.evaluateAsStringIfExpression(event, expression, name);
  }

  protected TypedValue evaluate(CoreEvent event, Object possibleExpression) {
    return expressionWrapper.evaluateIfExpression(event, possibleExpression);
  }

  private String evaluateMediaTypeExpression(CoreEvent event, Object mediaTypeExpression, MediaType valueMediaType) {
    if (mediaTypeExpression == null) {
      return valueMediaType.toRfcString();
    }
    return expressionWrapper.evaluateNotNullString(event, mediaTypeExpression, MEDIA_TYPE_FIELD);
  }

  public void setCloneOriginalEvent(Boolean cloneOriginalEvent) {
    this.cloneOriginalEvent = cloneOriginalEvent;
  }

  public void setPayload(Payload payload) {
    event.setPayload(payload);
  }

  public void setAttributes(EventAttributes attributes) {
    event.setAttributes(attributes);
  }

  public void setVariables(List<Variable> variables) {
    notNull(variables, "Variables can not be null");
    event.setVariables(variables);
  }

  public void setError(UntypedEventError error) {
    event.setError(error.toEventError());
  }

  protected void setMuleContext(PrivilegedMuleContext muleContext) {
    this.muleContext = muleContext;
  }

}
