/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.tools.mock;

import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.ArgumentMatchers.nullable;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.munit.common.api.model.Answer;
import org.mule.munit.common.api.model.Attribute;
import org.mule.munit.common.api.model.Event;
import org.mule.munit.common.api.model.EventError;
import org.mule.munit.common.api.model.FlowName;
import org.mule.munit.mock.MockModule;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.extension.api.runtime.process.RouterCompletionCallback;
import org.mule.runtime.extension.api.runtime.route.Chain;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;

import java.util.ArrayList;
import java.util.List;

import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;

public class MockOperationsTest {

  private static final String FLOW_NAME = "flow";
  private static final String PROCESSOR = "prefix:name";

  private Registry muleRegistryMock;
  private MockModule mockModuleMock;
  private StreamingHelper streamingHelperMock;

  private MockOperations mockOperations;

  private FlowName flowNameMock;
  private Answer answerMock;

  @Before
  public void setUp() {
    muleRegistryMock = mock(Registry.class);
    mockModuleMock = mock(MockModule.class);
    streamingHelperMock = mock(StreamingHelper.class);
    flowNameMock = mock(FlowName.class);
    answerMock = mock(Answer.class);

    when(flowNameMock.getFlow()).thenReturn(FLOW_NAME);
    when(answerMock.getThenReturn()).thenReturn(null);
    when(answerMock.getThenCall()).thenReturn(null);

    mockOperations = new MockOperations();
    mockOperations.setRegistry(muleRegistryMock);
    mockOperations.setMockModule(mockModuleMock);
  }

  @Test
  public void mockWhenNullThenReturn() {
    List<Attribute> withAttributes = new ArrayList<>();

    mockOperations.mockWhen(PROCESSOR, withAttributes, answerMock, streamingHelperMock);

    Mockito.verify(mockModuleMock, Mockito.times(1)).setRegistry(muleRegistryMock);
    Mockito.verify(mockModuleMock, Mockito.times(1)).when(PROCESSOR, withAttributes, (Event) null);
  }

  @Test
  public void mockWhenNullAttributes() {
    when(answerMock.getThenCall()).thenReturn(flowNameMock);

    mockOperations.mockWhen(PROCESSOR, null, answerMock, streamingHelperMock);

    Mockito.verify(mockModuleMock, Mockito.times(1)).setRegistry(muleRegistryMock);
    Mockito.verify(mockModuleMock, Mockito.times(1)).when(PROCESSOR, null, flowNameMock);
  }

  @Test(expected = IllegalArgumentException.class)
  public void mockWhenWithErrorIdAndCause() {
    List<Attribute> withAttributes = new ArrayList<>();

    Event eventMock = mock(Event.class);
    EventError eventErrorMock = mock(EventError.class);

    when(answerMock.getThenReturn()).thenReturn(eventMock);

    when(eventMock.getError()).thenReturn(eventErrorMock);
    when(eventErrorMock.getTypeId()).thenReturn("MULE:ANY");
    when(eventErrorMock.getCause()).thenReturn(new RuntimeException());

    mockOperations.mockWhen(PROCESSOR, withAttributes, answerMock, streamingHelperMock);
  }

  @Test
  public void mockWhen() {
    List<Attribute> withAttributes = new ArrayList<>();

    Event eventMock = mock(Event.class);
    when(answerMock.getThenReturn()).thenReturn(eventMock);

    mockOperations.mockWhen(PROCESSOR, withAttributes, answerMock, streamingHelperMock);

    Mockito.verify(mockModuleMock, Mockito.times(1)).setRegistry(muleRegistryMock);
    Mockito.verify(mockModuleMock, Mockito.times(1)).when(PROCESSOR, withAttributes, eventMock);
  }

  @Test
  public void verifyTimes() {
    List<Attribute> withAttributes = new ArrayList<>();
    Integer times = 1;
    Integer atLeast = null;
    Integer atMost = null;

    mockOperations.verifyCall(PROCESSOR, withAttributes, times, atLeast, atMost);

    Mockito.verify(mockModuleMock, Mockito.times(1)).setRegistry(muleRegistryMock);
    Mockito.verify(mockModuleMock, Mockito.times(1)).verifyCall(PROCESSOR, withAttributes, times, atLeast, atMost);
  }

  @Test
  public void verifyAtLeast() {
    List<Attribute> withAttributes = new ArrayList<>();
    Integer times = null;
    Integer atLeast = 1;
    Integer atMost = null;

    mockOperations.verifyCall(PROCESSOR, withAttributes, times, atLeast, atMost);

    Mockito.verify(mockModuleMock, Mockito.times(1)).setRegistry(muleRegistryMock);
    Mockito.verify(mockModuleMock, Mockito.times(1)).verifyCall(PROCESSOR, withAttributes, times, atLeast, atMost);
  }

  @Test
  public void verifyAtMost() {
    List<Attribute> withAttributes = new ArrayList<>();
    Integer times = null;
    Integer atLeast = null;
    Integer atMost = 1;

    mockOperations.verifyCall(PROCESSOR, withAttributes, times, atLeast, atMost);

    Mockito.verify(mockModuleMock, Mockito.times(1)).setRegistry(muleRegistryMock);
    Mockito.verify(mockModuleMock, Mockito.times(1)).verifyCall(PROCESSOR, withAttributes, times, atLeast, atMost);
  }

  @Test
  public void verifyNullAttributes() {
    List<Attribute> withAttributes = null;
    Integer times = null;
    Integer atLeast = null;
    Integer atMost = null;

    mockOperations.verifyCall(PROCESSOR, withAttributes, times, atLeast, atMost);

    Mockito.verify(mockModuleMock, Mockito.times(1)).setRegistry(muleRegistryMock);
    Mockito.verify(mockModuleMock, Mockito.times(1)).verifyCall(PROCESSOR, withAttributes, 1, atLeast, atMost);
  }

  @Test
  public void verify() {
    List<Attribute> withAttributes = new ArrayList<>();
    Integer times = null;
    Integer atLeast = null;
    Integer atMost = null;

    mockOperations.verifyCall(PROCESSOR, withAttributes, times, atLeast, atMost);

    Mockito.verify(mockModuleMock, Mockito.times(1)).setRegistry(muleRegistryMock);
    Mockito.verify(mockModuleMock, Mockito.times(1)).verifyCall(PROCESSOR, withAttributes, 1, atLeast, atMost);
  }

  @Test
  public void spyNullChains() {
    List<Attribute> withAttributes = new ArrayList<>();

    RouterCompletionCallback callbackMock = mock(RouterCompletionCallback.class);
    mockOperations.spy(PROCESSOR, withAttributes, null, null, callbackMock);

    Mockito.verify(mockModuleMock).setRegistry(muleRegistryMock);
    Mockito.verify(mockModuleMock).spy(PROCESSOR, withAttributes, null, null);
    Mockito.verify(callbackMock).success(any());
  }

  @Test
  public void spy() {
    List<Attribute> withAttributes = new ArrayList<>();
    BeforeCall beforeCallMock = mock(BeforeCall.class);
    AfterCall afterCallMock = mock(AfterCall.class);

    Chain beforeChain = mock(Chain.class);
    Chain afterChain = mock(Chain.class);

    when(beforeCallMock.getChain()).thenReturn(beforeChain);
    when(afterCallMock.getChain()).thenReturn(afterChain);

    RouterCompletionCallback callbackMock = mock(RouterCompletionCallback.class);
    mockOperations.spy(PROCESSOR, withAttributes, beforeCallMock, afterCallMock, callbackMock);

    Mockito.verify(mockModuleMock).setRegistry(muleRegistryMock);
    Mockito.verify(mockModuleMock).spy(PROCESSOR, withAttributes, beforeChain, afterChain);
    Mockito.verify(callbackMock).success(any());
  }

  @Test
  public void verifyAttributesValidation() {
    assertExceptionIsThrown(() -> mockOperations.verifyCall(PROCESSOR, null, 1, 1, 1), IllegalArgumentException.class);
    assertExceptionIsThrown(() -> mockOperations.verifyCall(PROCESSOR, null, 1, 1, null), IllegalArgumentException.class);
    assertExceptionIsThrown(() -> mockOperations.verifyCall(PROCESSOR, null, 1, null, 1), IllegalArgumentException.class);

    mockOperations.verifyCall(PROCESSOR, null, null, null, null);
    mockOperations.verifyCall(PROCESSOR, null, 1, null, null);
    mockOperations.verifyCall(PROCESSOR, null, null, 1, null);
    mockOperations.verifyCall(PROCESSOR, null, null, null, 1);
    mockOperations.verifyCall(PROCESSOR, null, null, 1, 1);

    Mockito.verify(mockModuleMock, Mockito.times(5)).setRegistry(muleRegistryMock);
    Mockito.verify(mockModuleMock, Mockito.times(5)).verifyCall(eq(PROCESSOR), nullable(List.class), nullable(Integer.class),
                                                                nullable(Integer.class), nullable(Integer.class));
  }

  private void assertExceptionIsThrown(Runnable runnable, Class<? extends Throwable> expected) {
    try {
      runnable.run();
      fail();
    } catch (Exception e) {
      assertTrue(expected.isInstance(e));
    }
  }
}
