/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.tools.util.store;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.fail;
import org.mule.functional.api.exception.ExpectedError;
import org.mule.munit.tools.assertion.MunitToolsTestCase;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.api.store.ObjectStoreManager;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.oauth.api.state.ResourceOwnerOAuthContext;

import java.util.Map;

import javax.inject.Inject;

import org.junit.Rule;
import org.junit.Test;

public class StorageTestCase extends MunitToolsTestCase {

  @Rule
  public ExpectedError expectedError = ExpectedError.none();

  @Inject
  private ObjectStoreManager objectStoreManager;

  @Override
  protected String getConfigFile() {
    return "util/data-storage.xml";
  }

  @Test
  public void storeAndRetrieve() throws Exception {
    Event result = runFlow("storeAndRetrieve");
    Integer value = (Integer) result.getMessage().getPayload().getValue();
    assertThat(value, equalTo(1));
  }

  @Test
  public void storeAndUpdateSuccess() throws Exception {
    Event result = runFlow("storeAndUpdateSuccess");
    Integer value = (Integer) result.getMessage().getPayload().getValue();
    assertThat(value, equalTo(2));
  }

  @Test
  public void storeAndUpdateFailsWhenNotAllowed() throws Exception {
    expectedError.expectErrorType("MUNIT-TOOLS", "KEY_ALREADY_EXISTS");
    runFlow("storeAndUpdateFailsWhenNotAllowed");
  }

  @Test
  public void storedValueIsNotModifiedWhenUpdateFails() throws Exception {
    try {
      runFlow("storeAndUpdateFailsWhenNotAllowed");
      fail("Flow should fail on update but result was success");
    } catch (Exception e) {
      CoreEvent result = flowRunner("retrieve").withVariable("key", "myKey").run();
      Integer value = (Integer) result.getMessage().getPayload().getValue();
      assertThat(value, equalTo(1));
    }
  }

  @Test
  public void retrieveMissingKey() throws Exception {
    expectMissingKey();
    runFlow("retrieveMissingKey");
  }

  @Test
  public void removeData() throws Exception {
    Event result = runFlow("removeData");
    Integer value = (Integer) result.getMessage().getPayload().getValue();
    assertThat(value, equalTo(1));
  }

  @Test
  public void removeMissingKey() throws Exception {
    expectMissingKey();
    runFlow("removeMissingKey");
  }

  @Test
  public void invalidKey() throws Exception {
    expectedError.expectErrorType("MUNIT-TOOLS", "INVALID_KEY");
    runFlow("invalidKey");
  }

  @Test
  public void clearData() throws Exception {
    expectMissingKey();
    runFlow("clearData");
  }

  @Test
  public void storeOAuthToken() throws Exception {
    doStoreToken(false);
  }

  @Test
  public void storeRepeatedOAuthToken() throws Exception {
    storeOAuthToken();

    expectKeyAlreadyExists();
    storeOAuthToken();
  }

  @Test
  public void overwriteOAuthToken() throws Exception {
    doStoreToken(true);
    doStoreToken(true);
  }

  private void doStoreToken(boolean overwrite) throws Exception {
    ObjectStore os = objectStoreManager.getObjectStore("tokenStore");
    assertThat(os, is(notNullValue()));

    flowRunner("storeOAuthToken").withVariable("overwrite", overwrite).run();

    ResourceOwnerOAuthContext ctx = (ResourceOwnerOAuthContext) os.retrieve("Cersei-Lannister");
    assertThat(ctx, is(notNullValue()));

    assertThat(ctx.getResourceOwnerId(), equalTo("Cersei"));
    assertThat(ctx.getAccessToken(), equalTo("firstOfHerName"));
    assertThat(ctx.getRefreshToken(), equalTo("queenOfTheAndals"));
    assertThat(ctx.getExpiresIn(), equalTo("never"));
    assertThat(ctx.getState(), equalTo("protectorOfTheRealm"));

    Map<String, Object> params = ctx.getTokenResponseParameters();
    assertThat(params.size(), is(2));
    assertThat(params.get("spouse"), equalTo("Robert"));
    assertThat(params.get("kids"), equalTo(3));
  }

  private void expectMissingKey() {
    expectedError.expectErrorType("MUNIT-TOOLS", "MISSING_KEY");
  }

  private void expectKeyAlreadyExists() {
    expectedError.expectErrorType("MUNIT-TOOLS", "KEY_ALREADY_EXISTS");
  }

}
