/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.ang.introspector;

import static org.mule.runtime.api.util.MuleSystemProperties.SYSTEM_PROPERTY_PREFIX;
import static org.mule.runtime.core.api.util.FileUtils.unzip;

import static java.lang.Boolean.getBoolean;
import static java.lang.String.format;
import static java.lang.System.lineSeparator;
import static java.nio.file.Files.createTempDirectory;
import static java.util.UUID.randomUUID;

import static org.yaml.snakeyaml.DumperOptions.FlowStyle.BLOCK;

import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.container.api.ModuleRepository;

import com.mulesoft.runtime.ang.classpath.ModuleRepositoryGenerator;
import com.mulesoft.runtime.ang.introspector.exception.IntrospectorException;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.nio.file.Path;
import java.util.Map;

import org.slf4j.Logger;
import org.yaml.snakeyaml.DumperOptions;
import org.yaml.snakeyaml.Yaml;

public class MuleApplicationAngFn {

  public static final String FORCE_CONFIG_PARSE_PROPERTY = SYSTEM_PROPERTY_PREFIX + "angis.forceConfigParse";

  private final Logger log;
  private final Path workdir;
  private final ModuleRepository moduleRepository;
  private final MuleArtifactAngDataExtractor muleArtifactAngDataExtractor;

  public MuleApplicationAngFn(Logger log) {
    this.log = log;
    String workDirPrefix = randomUUID().toString();
    try {
      this.workdir = createTempDirectory(workDirPrefix);
    } catch (IOException e) {
      throw new IntrospectorException(format("Unable to create work directory with prefix '%s'", workDirPrefix), e);
    }

    try {
      this.moduleRepository = new ModuleRepositoryGenerator().read();
    } catch (IOException e) {
      throw new IntrospectorException("Unable to read moduleRepository data.", e);
    }
    this.muleArtifactAngDataExtractor = new MuleArtifactAngDataExtractor();
  }

  public String handle(String appFileName, InputStream muleAppJarFileContent, Map<String, Object> baseAppData) {
    log.info("Processing artifact '{}'...", appFileName);

    File appFolder = new File(workdir.toFile(), appFileName + randomUUID().toString());

    ArtifactAst artifact;
    try {
      log.debug("Unzipping mule-application jar file...");
      unzip(muleAppJarFileContent, appFolder, false, false);

      log.debug("Loading artifact AST...");
      artifact = getMuleArtifactAstLoader().loadFromArtifact(appFolder);
    } catch (IOException e) {
      throw new IntrospectorException(format("Unable to unzip jar file '%s' into work directory '%s'", appFileName,
                                             appFolder.getAbsolutePath().toString()),
                                      e);
    }

    log.debug("Extracting data from AST...");
    baseAppData.putAll(getMuleArtifactAngDataExtractor().extractFrom(artifact));

    log.debug("Generating YAML with extracted data...");
    final StringWriter outputWriter = new StringWriter();
    outputWriter.write("#%Application 1.0" + lineSeparator() + lineSeparator());
    createYaml().dump(baseAppData, outputWriter);

    log.info("Finished processing artifact '{}'.", appFileName);
    return outputWriter.toString();
  }

  protected MuleArtifactAstLoader getMuleArtifactAstLoader() {
    // Create a new loader for this artifact because its inner state may not be thread safe to share among many Fn calls
    return new MuleArtifactAstLoader(moduleRepository, log, getBoolean(FORCE_CONFIG_PARSE_PROPERTY));
  }

  protected MuleArtifactAngDataExtractor getMuleArtifactAngDataExtractor() {
    return muleArtifactAngDataExtractor;
  }

  static Yaml createYaml() {
    DumperOptions options = new DumperOptions();
    options.setIndent(2);
    options.setPrettyFlow(true);
    options.setDefaultFlowStyle(BLOCK);

    return new Yaml(options);
  }

  public Path getWorkdir() {
    return workdir;
  }
}
