/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.ang.introspector;

import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;
import static org.mule.runtime.module.deployment.impl.internal.maven.AbstractMavenClassLoaderModelLoader.CLASSLOADER_MODEL_JSON_DESCRIPTOR_LOCATION;

import static java.lang.String.format;
import static java.lang.Thread.currentThread;
import static java.util.function.UnaryOperator.identity;
import static java.util.stream.Collectors.toMap;

import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.serialization.ArtifactAstDeserializer;
import org.mule.runtime.ast.api.serialization.ArtifactAstSerializerProvider;
import org.mule.runtime.ast.api.xml.AstXmlParser;
import org.mule.runtime.container.api.ModuleRepository;
import org.mule.runtime.dsl.api.ConfigResource;

import com.mulesoft.runtime.ang.introspector.extension.DefaultExtensionModelLoader;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;

/**
 * Loads an {@link ArtifactAst} from an exploded Mule application.
 */
public class MuleArtifactAstLoader {

  private static final String SERIALIZED_ARTIFACT_AST_LOCATION = "META-INF/mule-artifact/artifact.ast";

  private final ArtifactAstDeserializer astDeserializer;

  private final ModuleRepository moduleRepository;

  private final Logger log;

  private final boolean forceConfigParse;

  /**
   * 
   * @param moduleRepository
   * @param log
   * @param forceConfigParse if set to {@code true}, any serialized AST present in the artifact will be ignored and the AST will
   *                         be generated by parsing the config.
   */
  public MuleArtifactAstLoader(ModuleRepository moduleRepository, Logger log, boolean forceConfigParse) {
    this.moduleRepository = moduleRepository;
    this.astDeserializer = new ArtifactAstSerializerProvider().getDeserializer(false);
    this.log = log;
    this.forceConfigParse = forceConfigParse;
  }

  /**
   * Loads an {@link ArtifactAst} from an exploded Mule application.
   * 
   * @param explodedAppDir the directory where the muleApp is exploded
   * @return the loaded {@link ArtifactAst} for the Mule application.
   * 
   * @throws IllegalArgumentException
   * @throws IOException
   */
  public ArtifactAst loadFromArtifact(File explodedAppDir) throws IllegalArgumentException, IOException {
    File serializedAstFile = new File(explodedAppDir, SERIALIZED_ARTIFACT_AST_LOCATION);
    File classloaderModelFile = new File(explodedAppDir, CLASSLOADER_MODEL_JSON_DESCRIPTOR_LOCATION);

    if (!classloaderModelFile.exists()) {
      throw new IllegalArgumentException("Artifact does not contain a 'classloader-model.json' file. It must be a heavyweight package.");
    }

    return loadArtifactAst(explodedAppDir, serializedAstFile);
  }

  private ArtifactAst loadArtifactAst(File explodedAppDir, File serializedAstFile) throws IOException {
    try (DefaultExtensionModelLoader loader =
        new DefaultExtensionModelLoader(explodedAppDir, this.getClass().getClassLoader(), moduleRepository)) {
      Set<ExtensionModel> extensionModels = loader.obtainExtensionModels();

      if (!forceConfigParse && serializedAstFile.exists()) {
        log.info("Attempting to deserialize artifactAst present in artifact...");
        Map<String, ExtensionModel> extensionsByName =
            extensionModels.stream().collect(toMap(ExtensionModel::getName, identity()));
        try (FileInputStream artifactAstInputStream = new FileInputStream(serializedAstFile)) {
          return astDeserializer.deserialize(artifactAstInputStream, extensionsByName::get);
        }
      } else {
        if (serializedAstFile.exists()) {
          log.info("Ignoring serialized artifactAst present in artifact, parsing the config...");
        } else {
          log.info("Serialized artifactAst not present in artifact, parsing the config...");
        }
        // For older versions of the packager or if for some reason the serialized AST is not in the package, parse the app
        // config files.
        return withContextClassLoader(loader.getApplicationClassLoader(),
                                      () -> parseArtifactConfigFiles(explodedAppDir,
                                                                     loader.getApplicationDescriptor().getConfigResources(),
                                                                     extensionModels));
      }
    }
  }

  private ArtifactAst parseArtifactConfigFiles(File explodedAppDir, Set<String> configResources,
                                               Set<ExtensionModel> extensionModels) {
    AstXmlParser astXmlParser = AstXmlParser.builder()
        .withExtensionModels(extensionModels)
        .build();

    ClassLoader tccl = currentThread().getContextClassLoader();

    return astXmlParser.parse(configResources.stream()
        .map(configResource -> {
          try {
            return new ConfigResource(configResource, tccl.getResource(configResource));
          } catch (IOException e) {
            throw new IllegalStateException(format("Cannot load config file '%s'.", configResource), e);
          }
        })
        .toArray(ConfigResource[]::new));
  }

}
