/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.ang.introspector.extension;

import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.startIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.stopIfNeeded;
import static org.mule.runtime.core.api.util.boot.ExtensionLoaderUtils.lookupExtensionModelLoaders;
import static org.mule.runtime.deployment.model.api.artifact.extension.ExtensionModelLoaderRepository.getExtensionModelLoaderManager;

import static java.util.Collections.unmodifiableSet;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.deployment.model.api.artifact.extension.ExtensionDiscoveryRequest;
import org.mule.runtime.deployment.model.api.artifact.extension.ExtensionModelDiscoverer;
import org.mule.runtime.deployment.model.api.artifact.extension.ExtensionModelLoaderRepository;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;

import java.util.List;
import java.util.Set;

import org.slf4j.Logger;

public class DefaultExtensionModelService {

  private static final Logger LOGGER = getLogger(DefaultExtensionModelService.class);

  private final ExtensionModelDiscoverer extensionModelDiscoverer = new ExtensionModelDiscoverer();
  private final MuleArtifactResourcesRegistry muleArtifactResourcesRegistry;

  private final Set<ExtensionModel> runtimeExtensionModels;

  public DefaultExtensionModelService(MuleArtifactResourcesRegistry muleArtifactResourcesRegistry,
                                      Set<ExtensionModel> runtimeExtensionModels) {
    requireNonNull(muleArtifactResourcesRegistry, "muleArtifactResourcesRegistry cannot be null");

    this.muleArtifactResourcesRegistry = muleArtifactResourcesRegistry;
    this.runtimeExtensionModels = unmodifiableSet(runtimeExtensionModels);
  }

  public Set<ExtensionModel> loadExtensionsData(List<ArtifactClassLoader> artifactPluginClassLoaders) {
    ExtensionModelLoaderRepository extensionModelLoaderRepository = null;
    try {

      extensionModelLoaderRepository =
          getExtensionModelLoaderManager(muleArtifactResourcesRegistry.getContainerArtifactClassLoader(),
                                         () -> lookupExtensionModelLoaders().collect(toSet()));
      startIfNeeded(extensionModelLoaderRepository);
      return getLoadedExtensionsInformation(artifactPluginClassLoaders, extensionModelLoaderRepository);
    } catch (Exception e) {
      throw new MuleRuntimeException(e);
    } finally {
      if (extensionModelLoaderRepository != null) {
        try {
          stopIfNeeded(extensionModelLoaderRepository);
        } catch (MuleException e) {
          LOGGER.error("Exception stopping extensionModelLoaderRepository: " + e.getMessage(), e);
        }
      }
    }
  }

  private Set<ExtensionModel> getLoadedExtensionsInformation(List<ArtifactClassLoader> artifactPluginClassLoaders,
                                                             ExtensionModelLoaderRepository loaderRepository) {
    return extensionModelDiscoverer.discoverPluginsExtensionModels(ExtensionDiscoveryRequest.builder()
        .setLoaderRepository(loaderRepository)
        .setArtifactPlugins(artifactPluginClassLoaders.stream()
            .map(plugincl -> new Pair<ArtifactPluginDescriptor, ArtifactClassLoader>(plugincl
                .getArtifactDescriptor(), plugincl))
            .collect(toList()))
        .setParentArtifactExtensions(runtimeExtensionModels)
        .setParallelDiscovery(true)
        .setEnrichDescriptions(false)
        .build())
        .stream()
        .map(Pair::getSecond)
        .collect(toSet());
  }

}
