/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.ang.introspector.extractor;

import static java.util.regex.Pattern.compile;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.ast.api.util.MuleAstUtils.doOnParamComponents;

import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.BinaryType;
import org.mule.metadata.api.model.BooleanType;
import org.mule.metadata.api.model.DateTimeType;
import org.mule.metadata.api.model.DateType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.NullType;
import org.mule.metadata.api.model.NumberType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.api.model.TimeType;
import org.mule.metadata.api.model.VoidType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.internal.dsl.DslConstants;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class ConfigPropertiesDataExtractor {

  private static final String NAME_PARAM = "name";
  private static final String VALUE_PARAM = "value";

  public static final String CONFIG_PROPERTIES_KEY = "config-properties";

  private static final String TYPE_KEY = "type";
  private static final String DEFAULT_KEY = "default";

  private static final ComponentIdentifier GLOBAL_PROPERTY_IDENTIFIER = ComponentIdentifier.builder().name("global-property")
      .namespace(DslConstants.CORE_PREFIX)
      .namespaceUri(DslConstants.CORE_NAMESPACE).build();

  private static final Pattern CONFIG_PROPERTY_USAGE = compile("(?<!\\\\)\\$\\{([^}]+)}");

  /**
   * Extracts the config properties form the app.
   * <p>
   * A config prooperty has a type that is determined from where the property is used, and a default value that is provided via a
   * {@code global-property} element in the app itself.
   * 
   * @param artifact
   * @return
   */
  public Map<String, Object> extractFrom(ArtifactAst artifact) {
    final Map<String, Object> configPropertiesMap = new LinkedHashMap<>();

    artifact.recursiveStream()
        .forEach(c -> extractConfigPropertiesFrom(c, configPropertiesMap, obtainDefaultValues(artifact)));

    return configPropertiesMap;
  }

  private Map<String, String> obtainDefaultValues(ArtifactAst artifact) {
    Map<String, String> defaultValues = new HashMap<>();

    // TODO EE-7984 default values may be set in a domain as well.
    artifact.topLevelComponentsStream()
        .filter(c -> c.getIdentifier().equals(GLOBAL_PROPERTY_IDENTIFIER))
        .forEach(c -> {
          ComponentParameterAst valueParam = c.getParameter(DEFAULT_GROUP_NAME, VALUE_PARAM);
          if (valueParam.getRawValue() != null) {
            ComponentParameterAst nameParam = c.getParameter(DEFAULT_GROUP_NAME, NAME_PARAM);
            defaultValues.put(nameParam.getRawValue(), valueParam.getRawValue());
          }
        });
    return defaultValues;
  }

  private void extractConfigPropertiesFrom(ComponentAst component, Map<String, Object> configPropertiesMap,
                                           Map<String, String> defaultValues) {
    component.getParameters()
        .stream()
        .filter(p -> p.getRawValue() != null)
        .forEach(p -> {
          findPropertyUsages(p.getRawValue(), propName -> {
            Map<Object, Object> configPropertyMap = new LinkedHashMap<>();

            populateDataType(p, configPropertyMap);

            if (defaultValues.containsKey(propName)) {
              configPropertyMap.put(DEFAULT_KEY, defaultValues.get(propName));
            } else if (p.getModel().getDefaultValue() != null) {
              configPropertyMap.put(DEFAULT_KEY, p.getModel().getDefaultValue().toString());
            }

            configPropertiesMap.putIfAbsent(propName, configPropertyMap);
          });
          doOnParamComponents(p, c -> extractConfigPropertiesFrom(c, configPropertiesMap, defaultValues));
        });
  }

  void findPropertyUsages(String rawValue, Consumer<String> onProperty) {
    Matcher matcher = CONFIG_PROPERTY_USAGE.matcher(rawValue);
    while (matcher.find()) {
      onProperty.accept(matcher.group(1));
    }
  }

  private void populateDataType(ComponentParameterAst p, Map<Object, Object> configPropertyMap) {
    p.getModel().getType().accept(new MetadataTypeVisitor() {

      @Override
      public void visitBinaryType(BinaryType binaryType) {
        configPropertyMap.put(TYPE_KEY, "binary");
      }

      @Override
      public void visitBoolean(BooleanType booleanType) {
        configPropertyMap.put(TYPE_KEY, "boolean");
      }

      @Override
      public void visitDateTime(DateTimeType dateTimeType) {
        configPropertyMap.put(TYPE_KEY, "dateTime");
      }

      @Override
      public void visitDate(DateType dateType) {
        configPropertyMap.put(TYPE_KEY, "date");
      }

      @Override
      public void visitNull(NullType nullType) {
        configPropertyMap.put(TYPE_KEY, "null");
      }

      @Override
      public void visitVoid(VoidType voidType) {
        configPropertyMap.put(TYPE_KEY, "void");
      }

      @Override
      public void visitNumber(NumberType numberType) {
        configPropertyMap.put(TYPE_KEY, "number");
      }

      @Override
      public void visitString(StringType stringType) {
        Optional<EnumAnnotation> annotation = stringType.getAnnotation(EnumAnnotation.class);
        if (annotation.isPresent()) {
          configPropertyMap.put(TYPE_KEY, "enum");
          configPropertyMap.put("values", annotation.get().getValues());
        } else {
          configPropertyMap.put(TYPE_KEY, "string");
        }
      }

      @Override
      public void visitTime(TimeType timeType) {
        configPropertyMap.put(TYPE_KEY, "time");
      }

      @Override
      public void visitObject(ObjectType objectType) {
        configPropertyMap.put(TYPE_KEY, "object");
      }

      @Override
      public void visitArrayType(ArrayType arrayType) {
        configPropertyMap.put(TYPE_KEY, "array");
      }

      @Override
      protected void defaultVisit(MetadataType metadataType) {
        configPropertyMap.put(TYPE_KEY, "other");
      }
    });
  }

}
