/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.ang.introspector.extractor;

import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.FLOW;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.SOURCE;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.internal.dsl.DslConstants.CORE_NAMESPACE;
import static org.mule.runtime.internal.dsl.DslConstants.CORE_PREFIX;

import static java.util.Collections.emptyMap;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.Function;

public class SchedulersDataExtractor {

  private static final String DESCRIPTION_DOC_PARAM = "description";
  private static final String MAX_CONCURRENCY_PARAM = "maxConcurrency";
  private static final String SCHEDULING_STRATEGY_PARAM = "schedulingStrategy";
  private static final String FIXED_FREQUENCY_SCH_PARAM_VALUE = "fixed-frequency";
  private static final String START_DELAY_PARAM = "startDelay";
  private static final String FREQUENCY_PARAM = "frequency";
  private static final String TIME_UNIT_PARAM = "timeUnit";
  private static final String CRON_SCH_PARAM_VALUE = "cron";
  private static final String CRON_EXPRESSION_PARAM = "expression";
  private static final String TIME_ZONE_PARAM = "timeZone";

  public static final String SCHEDULERS_KEY = "schedulers";

  private static final String SCHEDULER_KEY = "scheduler";
  private static final String STRATEGY_KEY = "strategy";
  private static final String FIXED_FREQUENCY_KEY = "fixedFrequency";
  private static final String TIME_UNIT_KEY = TIME_UNIT_PARAM;
  private static final String START_DELAY_KEY = START_DELAY_PARAM;
  private static final String CRON_EXPRESSION_KEY = "cronExpression";
  private static final String TIME_ZONE_KEY = TIME_ZONE_PARAM;

  private static final ComponentIdentifier CRON_IDENTIFIER = ComponentIdentifier.builder()
      .namespaceUri(CORE_NAMESPACE)
      .namespace(CORE_PREFIX)
      .name(CRON_SCH_PARAM_VALUE).build();
  private static final ComponentIdentifier FIXED_FREQUENCY_IDENTIFIER = ComponentIdentifier.builder()
      .namespaceUri(CORE_NAMESPACE)
      .namespace(CORE_PREFIX)
      .name(FIXED_FREQUENCY_SCH_PARAM_VALUE).build();

  private final Map<ComponentIdentifier, Function<ComponentAst, Map<String, String>>> schedulerExtractors;

  public SchedulersDataExtractor() {
    schedulerExtractors = new HashMap<>();

    schedulerExtractors.put(FIXED_FREQUENCY_IDENTIFIER, this::extractFixedFrequencyStrategyParameters);
    schedulerExtractors.put(CRON_IDENTIFIER, this::extractCronStrategyParameters);
  }

  public Map<String, Object> extractFrom(ArtifactAst artifact) {
    final Map<String, Object> schedulersMap = new LinkedHashMap<>();

    artifact.topLevelComponentsStream()
        .filter(comp -> comp.getComponentType().equals(FLOW))
        .forEach(flow -> flow.directChildrenStream()
            .filter(comp -> comp.getComponentType().equals(SOURCE))
            .filter(src -> src.getParameter(DEFAULT_GROUP_NAME, SCHEDULING_STRATEGY_PARAM) != null)
            .forEach(src -> schedulersMap.put(flow.getComponentId().get(), extractFlowScheduler(flow, src))));

    return schedulersMap;
  }

  public Map<String, Object> extractFlowScheduler(ComponentAst flow, ComponentAst src) {
    Map<String, Object> flowOut = new LinkedHashMap<>();

    Map<String, String> docAttributes = flow.getMetadata().getDocAttributes();
    if (docAttributes.containsKey(DESCRIPTION_DOC_PARAM)) {
      flowOut.put(DESCRIPTION_DOC_PARAM, docAttributes.get(DESCRIPTION_DOC_PARAM));
    }

    putIfPresent(flowOut, MAX_CONCURRENCY_PARAM, flow.getParameter(DEFAULT_GROUP_NAME, MAX_CONCURRENCY_PARAM));

    // Still not clear if this flag goes into the ANG data
    // if (src.getModel(ParameterizedModel.class).get().getName().equals("scheduler")) {
    // // Only core schedulers have this parameter, extension polling sources do not.
    // src.getParameter(DEFAULT_GROUP_NAME, "disallowConcurrentExecution")
    // .getValue().applyRight(disallowConcurrentExecution -> {
    // if ((boolean) disallowConcurrentExecution) {
    // flowOut.put("disallowConcurrentExecution", TRUE);
    // }
    // });
    // }

    src.getParameter(DEFAULT_GROUP_NAME, SCHEDULING_STRATEGY_PARAM).getValue()
        .mapRight(sch -> {
          ComponentAst scheduler = (ComponentAst) sch;
          if (schedulerExtractors.containsKey(scheduler.getIdentifier())) {
            return schedulerExtractors.get(scheduler.getIdentifier()).apply(scheduler);
          } else {
            return emptyMap();
          }
        })
        .applyRight(schedulerOut -> flowOut.put(SCHEDULER_KEY, schedulerOut));

    return flowOut;
  }

  private Map<String, String> extractFixedFrequencyStrategyParameters(final ComponentAst scheduler) {
    Map<String, String> schedulerOut = new LinkedHashMap<>();

    schedulerOut.put(STRATEGY_KEY, FIXED_FREQUENCY_KEY);

    putIfPresent(schedulerOut, TIME_UNIT_KEY, scheduler.getParameter(FIXED_FREQUENCY_SCH_PARAM_VALUE, TIME_UNIT_PARAM));
    putIfPresent(schedulerOut, FIXED_FREQUENCY_KEY, scheduler.getParameter(FIXED_FREQUENCY_SCH_PARAM_VALUE, FREQUENCY_PARAM));
    putIfPresent(schedulerOut, START_DELAY_KEY, scheduler.getParameter(FIXED_FREQUENCY_SCH_PARAM_VALUE, START_DELAY_PARAM));

    return schedulerOut;
  }

  private Map<String, String> extractCronStrategyParameters(final ComponentAst scheduler) {
    Map<String, String> schedulerOut = new LinkedHashMap<>();

    schedulerOut.put(STRATEGY_KEY, CRON_SCH_PARAM_VALUE);

    putIfPresent(schedulerOut, CRON_EXPRESSION_KEY, scheduler.getParameter(CRON_SCH_PARAM_VALUE, CRON_EXPRESSION_PARAM));
    putIfPresent(schedulerOut, TIME_ZONE_KEY, scheduler.getParameter(CRON_SCH_PARAM_VALUE, TIME_ZONE_PARAM));

    return schedulerOut;
  }

  private <T> void putIfPresent(Map<String, T> out, String outParamName, ComponentParameterAst param) {
    if (param.getRawValue() != null) {
      out.put(outParamName, (T) param.getRawValue());
    } else if (param.getModel().getDefaultValue() != null) {
      out.put(outParamName, (T) param.getModel().getDefaultValue());
    }
  }

}
