/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.soap.message;

import org.apache.commons.io.IOUtils;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.extension.api.soap.SoapAttachment;

import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

import static java.util.Collections.emptyMap;
import static org.mule.soapkit.soap.util.Cast.cast;

public class SubFlowResponse {

  private static final String BODY = "body";
  private static final String HEADERS = "headers";
  private static final String ATTACHMENTS = "attachments";
  private static final String CONTENT_TYPE = "contentType";
  private static final String CONTENT = "content";

  private final InputStream body;
  private final Map<String, String> headers;
  private final Map<String, SoapAttachment> attachments;


  private SubFlowResponse(InputStream body, Map<String, String> headers,
                          Map<String, SoapAttachment> attachments) {
    this.body = body;
    this.headers = headers;
    this.attachments = attachments;
  }

  public InputStream getBody() {
    return body;
  }

  public Map<String, String> getHeaders() {
    return headers;
  }

  public Map<String, SoapAttachment> getAttachments() {
    return attachments;
  }

  public static SubFlowResponse create(Map map, Charset charset) {

    InputStream body = createBody(map.get(BODY), charset);
    Map<String, String> headers = createHeaders(map.get(HEADERS));
    Map<String, SoapAttachment> attachments = createAttachments(map.get(ATTACHMENTS));

    return new SubFlowResponse(body, headers, attachments);
  }

  public static SubFlowResponse createFromBody(InputStream body) {
    return new SubFlowResponse(body, Collections.emptyMap(), Collections.emptyMap());
  }

  private static InputStream createBody(Object value, Charset charset) {
    if (value instanceof InputStream) {
      return (InputStream) value;
    }
    if (value instanceof String) {
      return IOUtils.toInputStream((String) value, charset);
    }
    throw new IllegalArgumentException("Invalid 'body' detected in payload");
  }

  private static Map<String, String> createHeaders(Object value) {
    if (value == null) {
      return emptyMap();
    }
    if (value instanceof Map) {
      return cast(value);
    }
    throw new IllegalArgumentException("Invalid 'headers' detected in payload");
  }

  private static Map<String, SoapAttachment> createAttachments(Object value) {
    if (value == null) {
      return emptyMap();
    }
    if (value instanceof Map) {
      Map<Object, Object> attachments = cast(value);

      Map<String, SoapAttachment> result = new LinkedHashMap<>();

      attachments.forEach((key, object) -> {
        String attachmentName = (String) key;

        if (!(object instanceof Map)) {
          throw new IllegalArgumentException("Invalid 'attachments' detected in payload. Problems with attachment '"
              + attachmentName + "'");
        }
        Map<Object, Object> attachmentValue = cast(object);

        Object contentObject = attachmentValue.get(CONTENT);
        InputStream content = (contentObject instanceof InputStream) ? (InputStream) contentObject
            : (contentObject instanceof String) ? IOUtils.toInputStream((String) contentObject) : null;

        if (content == null) {
          throw new IllegalArgumentException("Invalid 'attachments' detected in payload. Problems with attachment '"
              + attachmentName + "'");
        }
        String contentType = (String) attachmentValue.get(CONTENT_TYPE);
        SoapAttachment soapResponseAttachment = new SoapAttachment(content, MediaType.parse(contentType));
        result.put(attachmentName, soapResponseAttachment);
      });

      return result;
    }
    throw new IllegalArgumentException("Invalid 'attachments' detected in payload");
  }

}
