/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.soap.server;

import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.message.Exchange;
import org.apache.cxf.service.invoker.Invoker;
import org.apache.cxf.service.model.BindingOperationInfo;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.extension.api.soap.SoapAttachment;
import org.mule.runtime.soap.api.exception.BadRequestException;
import org.mule.runtime.soap.api.message.SoapRequest;
import org.mule.runtime.soap.api.message.SoapRequestBuilder;
import org.mule.soapkit.soap.SoapConstants;
import org.mule.soapkit.soap.api.server.SoapServerHandler;
import org.mule.soapkit.soap.message.SoapResponse;
import org.mule.soapkit.soap.util.Cast;
import org.mule.soapkit.soap.util.XmlTransformationException;
import org.mule.soapkit.soap.util.XmlTransformationUtils;

import javax.xml.stream.XMLStreamReader;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.List;
import java.util.Map;

import static java.util.stream.Collectors.toMap;
import static org.apache.cxf.message.Message.CONTENT_TYPE;
import static org.apache.cxf.message.Message.PROTOCOL_HEADERS;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_XML;
import static org.mule.runtime.core.api.rx.Exceptions.rxExceptionToMuleException;

/**
 * Invokes a Mule Service via a CXF binding.
 */
class SoapCxfInvoker implements Invoker {

  SoapCxfInvoker() {}

  @Override
  public Object invoke(final Exchange exchange, final Object o) {
    try {
      final SoapServerHandler serverHandler = Cast.cast(exchange.get(SoapConstants.MULE_SERVER_HANDLER_KEY));

      // Get payload from Cxf
      final XMLStreamReader payload = extractPayload(exchange);

      // Get headers from Cxf
      final Map<String, String> soapHeader = extractHeaders(exchange);

      // Get attachments from Cxf
      final Map<String, SoapAttachment> attachments = extractAttachments(exchange);

      // Get transport headers from Cxf
      final Map<String, String> transportHeaders = extractTransportHeaders(exchange);

      final BindingOperationInfo bop = exchange.get(BindingOperationInfo.class);
      final String operation = bop.getOperationInfo().getName().getLocalPart();

      MediaType contentType = getContentType(transportHeaders);
      final SoapRequestBuilder builder = SoapRequest.builder()
          .operation(operation)
          .contentType(contentType)
          .soapHeaders(soapHeader)
          .attachments(attachments)
          .transportHeaders(transportHeaders)
          .content(xmlStreamReaderToInputStream(payload));

      try {
        final SoapResponse response = serverHandler.handle(builder.build());

        // Used by outbound Cxf interceptors (OutputSoapHeadersInterceptor, CopyAttachmentOutInterceptor)
        exchange.put(SoapConstants.SERVER_RESPONSE_KEY, response);

        // Used by OutputPayloadInterceptor
        return new Object[] {response};
      } catch (final Throwable throwable) {
        throw rxExceptionToMuleException(throwable);
      }
    } catch (Throwable t) {
      final Fault fault = findFaultException(t);
      if (fault != null) {
        throw fault;
      }
      throw new Fault(t);
    }
  }

  private MediaType getContentType(Map<String, String> transportHeaders) {
    Charset charset = transportHeaders.keySet().stream()
        .filter(CONTENT_TYPE::equalsIgnoreCase)
        .findAny()
        .map(transportHeaders::get)
        .flatMap(s -> MediaType.parse(s).getCharset())
        .orElseGet(Charset::defaultCharset);
    return APPLICATION_XML.withCharset(charset);
  }

  private Fault findFaultException(final Throwable e) {
    if (e == null)
      return null;
    if (e instanceof Fault)
      return (Fault) e;
    return findFaultException(e.getCause());
  }

  private Map<String, SoapAttachment> extractAttachments(final Exchange exchange) {
    return Cast.cast(exchange.get(SoapConstants.MULE_ATTACHMENTS_KEY));
  }

  private Map<String, String> extractTransportHeaders(final Exchange exchange) {
    final Map<String, List<String>> headers = Cast.cast(exchange.getInMessage().get(PROTOCOL_HEADERS));

    return headers.entrySet().stream()
        .collect(toMap(
                       Map.Entry::getKey, // Key
                       e -> e.getValue().isEmpty() ? "" : e.getValue().get(0))); // Value
  }

  private Map<String, String> extractHeaders(final Exchange exchange) {
    return Cast.cast(exchange.get(SoapConstants.MULE_HEADERS_KEY));
  }

  private XMLStreamReader extractPayload(final Exchange exchange) {
    return exchange.getInMessage().getContent(XMLStreamReader.class);
  }

  private InputStream xmlStreamReaderToInputStream(final XMLStreamReader reader) {
    try {
      return XmlTransformationUtils.xmlStreamReaderToInputStream(reader);
    } catch (final XmlTransformationException e) {
      throw new BadRequestException("Error transforming the soap request to be processed", e);
    }
  }
}
