/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.soap.server.interceptor;

import java.util.List;
import javax.xml.namespace.QName;
import org.apache.cxf.endpoint.Endpoint;
import org.apache.cxf.interceptor.AbstractInDatabindingInterceptor;
import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.message.Exchange;
import org.apache.cxf.message.Message;
import org.apache.cxf.phase.Phase;
import org.apache.cxf.service.model.BindingInfo;
import org.apache.cxf.service.model.BindingOperationInfo;
import org.apache.cxf.service.model.MessageInfo;
import org.apache.cxf.service.model.MessagePartInfo;
import org.apache.cxf.service.model.OperationInfo;
import org.apache.cxf.staxutils.DepthXMLStreamReader;
import org.apache.cxf.staxutils.StaxUtils;

public class SeekBopInterceptor extends AbstractInDatabindingInterceptor {

  public SeekBopInterceptor() {
    super(Phase.UNMARSHAL);
  }

  @Override
  public void handleMessage(Message message) throws Fault {
    DepthXMLStreamReader xmlReader = getXMLStreamReader(message);
    Exchange exchange = message.getExchange();
    BindingOperationInfo bop = exchange.getBindingOperationInfo();

    boolean client = isRequestor(message);

    //if body is empty and we have BindingOperationInfo, we do not need to match
    //operation anymore, just return
    if (bop != null && !StaxUtils.toNextElement(xmlReader)) {
      // body may be empty for partial response to decoupled request
      return;
    }
    bop = getBindingOperationInfo(xmlReader, exchange, bop, client);

    if (bop != null) {
      exchange.put(BindingOperationInfo.class, bop);
      exchange.put(OperationInfo.class, bop.getOperationInfo());
    }
  }

  private BindingOperationInfo getBindingOperationInfo(DepthXMLStreamReader xmlReader, Exchange exchange,
                                                       BindingOperationInfo bop, boolean client) {
    //bop might be a unwrapped, wrap it back so that we can get correct info
    if (bop != null && bop.isUnwrapped()) {
      bop = bop.getWrappedOperation();
    }

    if (bop == null) {
      QName startQName = xmlReader == null
          ? new QName("http://cxf.apache.org/jaxws/provider", "invoke")
          : xmlReader.getName();
      bop = getBindingOperationInfo(exchange, startQName, client);
    }
    return bop;
  }

  /**
  * We are overriding this method because we need call a custom getOperation because ServiceModelUtil.getOperation()
  * compare opName with name but we have to compare agains the Request Name
  */
  protected BindingOperationInfo getBindingOperationInfo(Exchange exchange, QName name, boolean client) {
    String local = name.getLocalPart();
    if (client && local.endsWith("Response")) {
      local = local.substring(0, local.length() - 8);
    }
    BindingOperationInfo bop = getOperation(exchange, local);
    if (bop != null) {
      exchange.put(BindingOperationInfo.class, bop);
      exchange.put(OperationInfo.class, bop.getOperationInfo());
    }
    return bop;
  }

  private static BindingOperationInfo getOperation(final Exchange exchange, final String opName) {
    final Endpoint ep = exchange.get(Endpoint.class);
    if (ep == null) {
      return null;
    }

    BindingInfo service = ep.getEndpointInfo().getBinding();

    for (BindingOperationInfo bopInfo : service.getOperations()) {
      MessageInfo messageInfo = bopInfo.getInput().getMessageInfo();
      List<MessagePartInfo> messageParts = messageInfo.getMessageParts();

      if (messageInfo.getName().getLocalPart().equals(opName)) {
        return bopInfo;
      }

      for (MessagePartInfo info : messageParts) {
        QName elementQName = info.getElementQName();
        String requestName = elementQName != null ? elementQName.getLocalPart() : "";
        if (requestName.equals(opName)) {
          return bopInfo;
        }
      }
    }
    return null;
  }
}
