/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.soap.util;

import com.ctc.wstx.api.WstxInputProperties;
import com.ctc.wstx.stax.WstxInputFactory;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.core.api.util.xmlsecurity.XMLSecureFactories;
import org.mule.runtime.soap.api.SoapService;
import org.mule.soapkit.soap.xml.stax.StaxSource;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.InputSource;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamReader;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.nio.charset.Charset;

import static com.ctc.wstx.api.WstxInputProperties.P_ALLOW_XML11_ESCAPED_CHARS_IN_XML10;
import static javax.xml.stream.XMLInputFactory.IS_COALESCING;

/**
 * {@link SoapService} Transformation utility class
 *
 * @since 1.0
 */
public class XmlTransformationUtils {

  private static final XMLInputFactory XML_INPUT_FACTORY = getXmlInputFactory();
  private static final TransformerFactory TRANSFORMER_FACTORY = XMLSecureFactories.createDefault().getTransformerFactory();

  private static XMLInputFactory getXmlInputFactory() {
    XMLInputFactory xmlInputFactory = WstxInputFactory.newInstance();
    // Preserve the CDATA tags
    xmlInputFactory.setProperty(IS_COALESCING, false);
    // This disables DTDs entirely for that factory
    xmlInputFactory.setProperty(XMLInputFactory.SUPPORT_DTD, false);
    xmlInputFactory.setProperty(P_ALLOW_XML11_ESCAPED_CHARS_IN_XML10, true);
    // disable external entities
    xmlInputFactory.setProperty("javax.xml.stream.isSupportingExternalEntities", false);
    return xmlInputFactory;
  }

  public static Element stringToDomElement(String xml) throws XmlTransformationException {
    try {
      DocumentBuilder db = XMLSecureFactories.createDefault().getDocumentBuilderFactory().newDocumentBuilder();
      InputSource is = new InputSource();
      is.setCharacterStream(new StringReader(xml));
      return db.parse(is).getDocumentElement();
    } catch (Exception e) {
      throw new XmlTransformationException("Could not transform xml string to Dom Element", e);
    }
  }

  public static String nodeToString(Node node) throws XmlTransformationException {
    try {
      StringWriter writer = new StringWriter();
      DOMSource source = new DOMSource(node);
      StreamResult result = new StreamResult(writer);
      Transformer transformer = TRANSFORMER_FACTORY.newTransformer();
      transformer.transform(source, result);
      return writer.toString();
    } catch (Exception e) {
      throw new XmlTransformationException("Could not transform Node to String", e);
    }
  }

  public static XMLStreamReader inputStreamToXmlStreamReader(InputStream inputStream, MediaType mediaType)
      throws XmlTransformationException {
    try {
      Charset charset = mediaType.getCharset().orElse(Charset.defaultCharset());
      return XML_INPUT_FACTORY
          .createXMLStreamReader(inputStream, charset.name());
    } catch (Exception e) {
      throw new XmlTransformationException("Could not transform xml to XmlStreamReader", e);
    }
  }

  public static InputStream xmlStreamReaderToInputStream(XMLStreamReader xmlStreamReader) throws XmlTransformationException {
    try {
      ByteArrayOutputStream outputStream = new ByteArrayOutputStream();

      final StaxSource source = new StaxSource(xmlStreamReader);
      final StreamResult result = new StreamResult(outputStream);
      Transformer transformer = TRANSFORMER_FACTORY.newTransformer();
      transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
      transformer.setOutputProperty(OutputKeys.ENCODING, xmlStreamReader.getEncoding());
      transformer.transform(source, result);
      return new ByteArrayInputStream(outputStream.toByteArray());
    } catch (TransformerException e) {
      throw new XmlTransformationException("Error transforming reader to DOM document", e);
    }
  }

}
