/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.soap.api.server;

import java.net.URL;
import java.util.Map;
import javax.xml.namespace.QName;
import org.mule.soap.api.SoapVersion;
import org.mule.soapkit.soap.api.param.ValidationLevel;
import org.mule.wsdl.parser.WsdlParser;
import org.mule.wsdl.parser.model.PortModel;
import org.mule.wsdl.parser.model.ServiceModel;
import org.mule.wsdl.parser.model.WsdlModel;

import static java.util.Objects.requireNonNull;

/**
 * Builder pattern implementation that creates {@link SoapServerConfiguration} instances.
 *
 * @since 4.1
 */
public class SoapServerConfigurationBuilder {

  private URL wsdlLocation;
  private String service;
  private String port;
  private SoapVersion version = null;
  private boolean mtomEnabled;
  private boolean validationEnabled = false;
  private boolean responseXMLDeclarationEnabled = false;
  private ValidationLevel validationLevel;
  private boolean payloadBody = true;
  private Map<String, String> namespaces;

  SoapServerConfigurationBuilder() {}

  /**
   * Sets the service of the WSDL we want to perform operations from.
   *
   * @return this builder.
   */
  public SoapServerConfigurationBuilder withService(final String service) {
    this.service = service;
    return this;
  }

  /**
   * Sets the port of the service that describes the set of operations that can be performed.
   *
   * @return this builder.
   */
  public SoapServerConfigurationBuilder withPort(final String port) {
    this.port = port;
    return this;
  }

  /**
  * Sets the {@link SoapVersion} of the Web Service. defaults to SOAP 1.1
  *
  * @return this builder.
  */
  public SoapServerConfigurationBuilder withVersion(final SoapVersion version) {
    this.version = version;
    return this;
  }

  /**
   * Specifies that the Web Service is MTOM enabled.
   *
   * @return this builder.
   */
  public SoapServerConfigurationBuilder enableMtom(boolean mtomEnabled) {
    this.mtomEnabled = mtomEnabled;
    return this;
  }

  /**
  * Specifies that the Web Service has inbound validation enabled.
  *
  * @return this builder.
  */
  public SoapServerConfigurationBuilder enableValidation(boolean validationEnabled) {
    this.validationEnabled = validationEnabled;
    return this;
  }

  /**
   * Specifies that the Web Service has response XML Declaration enabled.
   *
   * @return this builder.
   */
  public SoapServerConfigurationBuilder enableResponseXMLDeclaration(boolean responseXMLDeclarationEnabled) {
    this.responseXMLDeclarationEnabled = responseXMLDeclarationEnabled;
    return this;
  }

  /**
   * Sets the location of the WSDL that describes the SOAP web service. This can be an URL to an
   * external resource (e.g http://somewsdl.com/hit?wsdl), just a reference to a local file or an application resource name.
   *
   * @return this builder.
   */
  public SoapServerConfigurationBuilder withWsdlLocation(final URL wsdlLocation) {
    this.wsdlLocation = wsdlLocation;
    return this;
  }

  public SoapServerConfigurationBuilder withNamespaces(Map<String, String> namespaces) {
    this.namespaces = namespaces;
    return this;
  }

  public SoapServerConfigurationBuilder withValidationErrorLevel(ValidationLevel validationLevel) {
    this.validationLevel = validationLevel;
    return this;
  }

  /**
  * @return a new {@link SoapServerConfiguration} instance with the attributes specified.
  */
  public SoapServerConfiguration build() {
    requireNonNull(wsdlLocation, "WSDL location cannot be null");
    requireNonNull(service, "Service cannot be null");
    requireNonNull(port, "Port cannot be null");

    final WsdlModel wsdlModel = WsdlParser.Companion.parse(wsdlLocation.toString());
    final ServiceModel serviceModel = wsdlModel.getService(this.service);
    requireNonNull(serviceModel, String.format("Service '%s' not found in '%s'", service, wsdlLocation));
    final PortModel portModel = serviceModel.getPort(port);
    requireNonNull(portModel, String.format("Port '%s' not found in '%s'", port, wsdlLocation));

    final QName qName = serviceModel.getQName();
    requireNonNull(qName, String.format("QName not found for Service '%s' in '%s'", service, wsdlLocation));

    final String namespace = qName.getNamespaceURI();
    requireNonNull(namespace, String.format("Namespace not found for Service '%s' in '%s'", service, wsdlLocation));

    return new SoapServerConfiguration(wsdlModel, wsdlLocation, namespace, serviceModel,
                                       portModel, version, mtomEnabled, validationEnabled, payloadBody, namespaces,
                                       validationLevel, responseXMLDeclarationEnabled);
  }
}
