/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.soap.message;

import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.soap.api.message.SoapAttachment;

import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Map;

import static java.util.Collections.unmodifiableMap;
import static org.mule.soapkit.soap.util.Cast.cast;

/**
 * Immutable {@link SoapResponse} implementation.
 *
 * @since 1.0
 */
public final class ImmutableSoapResponse extends AbstractSoapResponse {

  private final InputStream content;
  private final Map<String, String> soapHeaders;
  private final Map<String, SoapAttachment> attachments;
  private final MediaType contentType;
  private Event event;

  public ImmutableSoapResponse(InputStream content,
                               Map<String, String> soapHeaders,
                               Map<String, String> transportHeaders,
                               Map<String, String> transportAdditionalData,
                               Map<String, SoapAttachment> attachments,
                               MediaType contentType,
                               Map<String, TypedValue<?>> variables) {
    super(transportHeaders, transportAdditionalData, variables);
    this.content = content;
    this.soapHeaders = unmodifiableMap(soapHeaders);
    this.attachments = unmodifiableMap(attachments);
    this.contentType = contentType;
  }

  public ImmutableSoapResponse(Event event,
                               Map<String, String> transportHeaders,
                               Map<String, String> transportAdditionalData,
                               MediaType contentType) {
    super(transportHeaders, transportAdditionalData, event.getVariables());
    // don't use event! this reference is set, to avoid that Mule Garbage Collector collect event reference
    this.event = event;
    SubFlowResponse payload = handlePayload(event.getMessage().getPayload(), contentType);
    this.content = payload.getBody();
    this.soapHeaders = unmodifiableMap(payload.getHeaders());
    this.attachments = unmodifiableMap(payload.getAttachments());
    this.contentType = contentType;
  }

  @Override
  public InputStream getContent() {
    return content;
  }

  @Override
  public Map<String, String> getSoapHeaders() {
    return soapHeaders;
  }

  @Override
  public Map<String, SoapAttachment> getAttachments() {
    return attachments;
  }

  @Override
  public String getContentType() {
    return contentType.toString();
  }

  private SubFlowResponse handlePayload(TypedValue<Object> payload, MediaType mediaType) {
    final Object value = payload.getValue();

    if (value instanceof Map) {
      return SubFlowResponse.create(cast(value), mediaType.getCharset().orElse(Charset.defaultCharset()));
    }
    if (value instanceof CursorStreamProvider) {
      return SubFlowResponse.createFromBody(((CursorStreamProvider) value).openCursor());
    }
    if (value instanceof InputStream) {
      return SubFlowResponse.createFromBody((InputStream) value);
    }
    throw new IllegalArgumentException("Invalid payload detected");
  }
}
