/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.soap.server;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.wsdl.WSDLException;
import javax.xml.namespace.QName;
import javax.xml.transform.Source;
import org.apache.cxf.common.logging.LogUtils;
import org.apache.cxf.service.factory.ServiceConstructionException;
import org.apache.cxf.service.model.EndpointInfo;
import org.apache.cxf.service.model.OperationInfo;
import org.apache.cxf.service.model.ServiceInfo;
import org.apache.cxf.wsdl.service.factory.ReflectionServiceFactoryBean;
import org.mule.runtime.api.i18n.I18nMessage;
import org.mule.soapkit.soap.api.server.SoapServerConfiguration;

class ProxyServiceFactoryBean extends ReflectionServiceFactoryBean {

  private static final String WSDL_LOCATION = "WSDL_LOCATION";
  private static final String INVOKE = "invoke";
  private static final Logger LOG = LogUtils.getLogger(ProxyServiceFactoryBean.class);
  private static ProxyServiceConfiguration proxyServiceConfiguration;

  ProxyServiceFactoryBean(SoapServerConfiguration config) {
    setWsdlURL(config.getWsdlLocation().toString());
    proxyServiceConfiguration = new ProxyServiceConfiguration(config);
    getServiceConfigurations().add(0, proxyServiceConfiguration);

    List<String> ignoredClasses = new ArrayList<>();
    ignoredClasses.add("java.lang.Object");
    ignoredClasses.add("java.lang.Throwable");
    ignoredClasses.add("org.omg.CORBA_2_3.portable.ObjectImpl");
    ignoredClasses.add("org.omg.CORBA.portable.ObjectImpl");
    ignoredClasses.add("javax.ejb.EJBObject");
    ignoredClasses.add("javax.rmi.CORBA.Stub");
    setIgnoredClasses(ignoredClasses);
  }

  @Override
  protected void initializeWSDLOperations() {
    if (getServiceClass().isAssignableFrom(ProxyService.class)) {
      initializeWSDLOperationsForProvider();
    } else {
      super.initializeWSDLOperations();
    }
  }

  private void initializeWSDLOperationsForProvider() {
    Class c = Source.class;

    if (getEndpointInfo() == null && isFromWsdl()) {
      // most likely, they specified a WSDL, but for some reason
      // did not give a valid ServiceName/PortName. For provider,
      // we'll allow this since everything is bound directly to
      // the invoke method, however, this CAN cause other problems
      // such as addresses in the wsdl not getting updated and such
      // so we'll WARN about it.....
      List<QName> enames = new ArrayList<>();
      for (ServiceInfo si : getService().getServiceInfos()) {
        for (EndpointInfo ep : si.getEndpoints()) {
          enames.add(ep.getName());
        }
      }
      I18nMessage i18nMessage = SoapServerMessages.couldNotFindEndpoint(getEndpointName(), enames);
      LOG.log(Level.WARNING, i18nMessage.getMessage(),
              new WSDLException(String.valueOf(i18nMessage.getCode()), i18nMessage.getMessage()));
    }

    try {
      getEndpointInfo().getService().setProperty(WSDL_LOCATION, getWsdlURL());

      final Method invoke = getServiceClass().getMethod(INVOKE, c);

      // Bind every name to the invoke method.
      for (ServiceInfo si : getService().getServiceInfos()) {
        for (OperationInfo o : si.getInterface().getOperations()) {
          getMethodDispatcher().bind(o, invoke);
        }
      }
    } catch (SecurityException | NoSuchMethodException e) {
      throw new ServiceConstructionException(e);
    }

  }

}
