/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.soap.message;


import static com.google.common.collect.ImmutableMap.copyOf;
import static com.google.common.collect.ImmutableMap.of;
import static java.util.stream.Collectors.joining;

import java.util.Map;


/**
 * The attributes returned by the consume operation, it carries the protocol specific headers (i.e. HTTP headers) and
 * additional transport data (for example HTTP status line) bounded to the response.
 *
 * @since 1.1.2
 */
public class SoapAttributes {

  private static final long serialVersionUID = 4591210489306615571L;

  /**
   * The protocol headers bundled in the response.
   */
  private final Map<String, String> protocolHeaders;

  /**
   * The additional transport data bundled in the response.
   */
  private final Map<String, String> additionalTransportData;

  public SoapAttributes(Map<String, String> protocolHeaders, Map<String, String> additionalTransportData) {
    this.protocolHeaders = protocolHeaders != null ? copyOf(protocolHeaders) : of();
    this.additionalTransportData = additionalTransportData != null ? copyOf(additionalTransportData) : of();
  }

  /**
   * @return a set of protocol headers bounded to the service response. i.e. HTTP Headers.
   */
  public Map<String, String> getProtocolHeaders() {
    return protocolHeaders;
  }

  /**
   * @return a set of additional transport data bounded to the service response (for example HTTP status line).
   */
  public Map<String, String> getAdditionalTransportData() {
    return additionalTransportData;
  }

  @Override
  public String toString() {
    String headersAsString = protocolHeaders.entrySet()
        .stream()
        .map(e -> e.getKey() + ":" + e.getValue())
        .collect(joining(",\n    "));

    String transportDataAsString = additionalTransportData.entrySet()
        .stream()
        .map(e -> e.getKey() + ":" + e.getValue())
        .collect(joining(",\n    "));

    return "{\n" +
        "  additionalTransportData = [\n" +
        "    " + transportDataAsString + "\n" +
        "  ]\n" +
        "  protocolHeaders = [\n" +
        "    " + headersAsString + "\n" +
        "  ]\n" +
        "}";
  }
}
