/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.soap.server;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import javax.wsdl.Definition;
import javax.wsdl.Port;
import javax.wsdl.Service;
import javax.wsdl.WSDLException;
import javax.xml.namespace.QName;
import org.apache.cxf.common.i18n.Message;
import org.apache.cxf.common.logging.LogUtils;
import org.apache.cxf.service.factory.ServiceConstructionException;
import org.apache.cxf.wsdl.WSDLManager;
import org.apache.cxf.wsdl.service.factory.DefaultServiceConfiguration;
import org.mule.runtime.api.i18n.I18nMessage;
import org.mule.soap.api.SoapVersion;
import org.mule.soapkit.soap.api.server.SoapServerConfiguration;

import static java.lang.String.format;
import static org.apache.commons.collections.CollectionUtils.select;

class ProxyServiceConfiguration extends DefaultServiceConfiguration {

  private static final Logger LOG = LogUtils.getLogger(ProxyServiceConfiguration.class);

  private SoapVersion soapVersion;

  private String portName;

  public ProxyServiceConfiguration(SoapServerConfiguration config) {
    this.portName = config.getPort();
    this.soapVersion = config.getVersion();
  }

  /**
   * Override to use port name from service definition in WSDL when we are doing WSDL-first. This is required so that CXF's
   * internal endpointName and port name match and a CXF Service gets created. See: https://issues.apache.org/jira/browse/CXF-1920
   * http://fisheye6.atlassian.com/changelog/cxf?cs=737994
   */
  @Override
  public QName getEndpointName() {
    try {
      if (getServiceFactory().getWsdlURL() != null) {
        Definition definition =
            getServiceFactory().getBus().getExtension(WSDLManager.class).getDefinition(getServiceFactory().getWsdlURL());
        Service service = getServiceFromDefinition(definition);
        setServiceNamespace(service.getQName().getNamespaceURI());


        QName endpointName;
        if (portName != null)
          endpointName = new QName(getServiceNamespace(), portName);
        else
          endpointName = new QName(getServiceNamespace(), ((Port) service.getPorts().values().iterator().next()).getName());


        LOG.fine(format("ProxyServiceConfiguration using endpoint %s", endpointName));
        return endpointName;
      } else {
        return super.getEndpointName();
      }

    } catch (WSDLException e) {
      throw new ServiceConstructionException(new Message("SERVICE_CREATION_MSG", LOG), e);
    }
  }


  private Service getServiceFromDefinition(Definition definition) throws WSDLException {
    Service service = definition.getService(getServiceFactory().getServiceQName());
    if (service == null) {
      List<QName> probableServices = getProbableServices(definition);
      List<QName> allServices = getAllServices(definition);

      I18nMessage message = SoapServerMessages.invalidOrMissingNamespace(getServiceFactory().getServiceQName(),
                                                                         probableServices, allServices);
      throw new WSDLException(String.valueOf(message.getCode()), message.getMessage());
    }
    return service;
  }

  /**
   * This method returns a list of all the services defined in the definition. Its current purpose is only for generating a better
   * error message when the service cannot be found.
   */
  @SuppressWarnings("unchecked")
  private List<QName> getAllServices(Definition definition) {
    return new LinkedList<QName>(select(definition.getServices().keySet(), object -> object instanceof QName));
  }

  /**
   * This method returns the list of services that matches with the local part of the service QName. Its current purpose is only
   * for generating a better error message when the service cannot be found.
   */
  private List<QName> getProbableServices(Definition definition) {
    QName serviceQName = getServiceFactory().getServiceQName();
    List<QName> probableServices = new LinkedList<QName>();
    Map<?, ?> services = definition.getServices();
    for (Object key : services.keySet()) {
      if (key instanceof QName) {
        QName qNameKey = (QName) key;
        if (qNameKey.getLocalPart() != null && qNameKey.getLocalPart().equals(serviceQName.getLocalPart())) {
          probableServices.add(qNameKey);
        }
      }
    }
    return probableServices;
  }
}
