/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.soap.server.interceptor;

import com.google.common.collect.ImmutableMap;
import org.apache.cxf.binding.soap.SoapHeader;
import org.apache.cxf.binding.soap.SoapMessage;
import org.apache.cxf.binding.soap.interceptor.AbstractSoapInterceptor;
import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.phase.Phase;
import org.mule.soap.api.client.BadRequestException;
import org.mule.soapkit.soap.util.XmlTransformationException;
import org.w3c.dom.Node;

import static java.lang.String.format;
import static org.mule.soapkit.soap.SoapConstants.MULE_HEADERS_KEY;
import static org.mule.soapkit.soap.util.XmlTransformationUtils.nodeToString;

public class InputSoapHeadersInterceptor extends AbstractSoapInterceptor {

  public InputSoapHeadersInterceptor() {
    super(Phase.PRE_PROTOCOL);
  }

  @Override
  public void handleMessage(SoapMessage message) throws Fault {

    final ImmutableMap.Builder<String, String> headers = ImmutableMap.builder();

    message.getHeaders().stream()
        .filter(header -> header instanceof SoapHeader)
        .map(h -> (SoapHeader) h)
        .forEach(header -> {
          headers.put(header.getName().getLocalPart(), getHeaderInputStream(header));
        });

    message.getExchange().put(MULE_HEADERS_KEY, headers.build());
  }

  private String getHeaderInputStream(SoapHeader header) {
    try {
      return nodeToString((Node) header.getObject());
    } catch (XmlTransformationException e) {
      throw new BadRequestException(format("Cannot parse content of SOAP header %s in the response",
                                           header.getName().getLocalPart()),
                                    e.getCause());
    }
  }

}
