/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.soap.server.interceptor;

import com.ctc.wstx.api.WstxOutputProperties;
import com.ctc.wstx.sw.BaseNsStreamWriter;
import org.apache.cxf.databinding.stax.XMLStreamWriterCallback;
import org.apache.cxf.interceptor.AbstractOutDatabindingInterceptor;
import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.message.MessageContentsList;
import org.apache.cxf.message.MessageUtils;
import org.apache.cxf.phase.Phase;
import org.apache.cxf.service.model.BindingOperationInfo;
import org.apache.cxf.service.model.MessagePartInfo;
import org.apache.cxf.staxutils.StaxUtils;
import org.mule.runtime.core.api.transformer.TransformerException;
import org.mule.soapkit.soap.message.SoapResponse;
import org.mule.soapkit.soap.util.XmlTransformationUtils;
import org.mule.runtime.api.metadata.MediaType;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;
import java.io.InputStream;
import java.util.LinkedList;
import java.util.List;

import static java.lang.Boolean.parseBoolean;
import static java.lang.System.getProperty;

public class OutputPayloadInterceptor
    extends AbstractOutDatabindingInterceptor {

  public static final String MULE_SOAPKIT_ALLOW_CTRL_CHARS = "mule.soapkit.allowControlCharacters";

  public OutputPayloadInterceptor() {
    super(Phase.PRE_LOGICAL);
  }

  @Override
  @SuppressWarnings("unchecked")
  public void handleMessage(org.apache.cxf.message.Message message) throws Fault {

    final MessageContentsList originalList = MessageContentsList.getContentsList(message);
    if (originalList == null || originalList.size() == 0)
      return;
    // why clone & clear original ???? Why more than one??
    final List<Object> clonedList = (List<Object>) originalList.clone();
    originalList.clear();

    for (final Object object : clonedList) {

      if (object instanceof SoapResponse)
        try {
          final SoapResponse response = (SoapResponse) object;
          final InputStream payload = response.getContent();
          InputStream inputStream = allowControlCharacters() ? new CtrlToNCRInputStream(payload) : payload;
          final XMLStreamReader streamReader =

              XmlTransformationUtils.inputStreamToXmlStreamReader(inputStream, MediaType.parse(response.getContentType()));
          XMLStreamWriterCallback xmlStreamWriterCallback = createXMLStreamWriterCallback(streamReader);
          originalList.add(xmlStreamWriterCallback);

        } catch (Exception e) {
          throw new Fault(e);
        }
      else {
        // it's probably a null object
        originalList.add(object);
      }
    }

    // For the server side response, ensure that the body object is in the correct
    // location when running in proxy mode
    if (!MessageUtils.isRequestor(message)) {
      BindingOperationInfo bop = message.getExchange().get(BindingOperationInfo.class);
      if (bop != null) {
        ensurePartIndexMatchListIndex(originalList, bop.getOutput().getMessageParts());
      }
    }
  }

  private XMLStreamWriterCallback createXMLStreamWriterCallback(final Object payload) throws TransformerException {
    final XMLStreamReader reader = (XMLStreamReader) payload;

    return new XMLStreamWriterCallback() {

      @Override
      public void write(XMLStreamWriter writer) throws Fault, XMLStreamException {
        if (allowControlCharacters()) {
          ((BaseNsStreamWriter) writer).setProperty(WstxOutputProperties.P_OUTPUT_INVALID_CHAR_HANDLER,
                                                    new NCRToControlCharHandler());
        }
        StaxUtils.copy(reader, writer);
        writer.flush();
        reader.close();
      }
    };
  }

  /**
   * Ensures that each part's content is in the right place in the content list.
   * <p/>
   * This is required because in some scenarios there are parts that were removed from the part list. In that cases, the content
   * list contains only the values for the remaining parts, but the part's indexes could be wrong. This method fixes that adding
   * null values into the content list so the part's index matches the contentList index. (Related to: MULE-5113.)
   */
  private void ensurePartIndexMatchListIndex(MessageContentsList contentList, List<MessagePartInfo> parts) {

    // In some circumstances, parts is a {@link UnmodifiableList} instance, so a new copy
    // is required in order to sort its content.
    List<MessagePartInfo> sortedParts = new LinkedList<>();
    sortedParts.addAll(parts);
    sortPartsByIndex(sortedParts);

    int currentIndex = 0;

    for (MessagePartInfo part : sortedParts) {
      while (part.getIndex() > currentIndex) {
        contentList.add(currentIndex++, null);
      }

      // Skips the index for the current part because now is in the right place
      currentIndex = part.getIndex() + 1;
    }
  }

  private void sortPartsByIndex(List<MessagePartInfo> parts) {
    parts.sort((o1, o2) -> o1.getIndex() < o2.getIndex() ? -1 : o1.getIndex() == o2.getIndex() ? 0 : 1);
  }

  private static boolean allowControlCharacters() {
    String allowControlCharactersInPayload = getProperty(MULE_SOAPKIT_ALLOW_CTRL_CHARS);
    if (allowControlCharactersInPayload == null)
      return false;
    return parseBoolean(allowControlCharactersInPayload);
  }
}
