package org.mule.weave.v2.module.commons.java.value

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.AlreadyMaterializedObjectSeq
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.LazyObjectSeq
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.structure.SimpleObjectSeq
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.module.commons.java.JavaValueHelper

import java.util
import scala.collection.JavaConverters._
import scala.collection.mutable

class JavaMapObjectValue(javaMap: java.util.Map[Any, Any], val locationString: () => String, converter: JavaValueConverter) extends JavaObjectValue {

  private lazy val value = JavaMapObjectSeq(javaMap, locationString, converter)

  override def evaluate(implicit ctx: EvaluationContext): T = {
    value
  }

  override def underlying()(implicit ctx: EvaluationContext): Any = javaMap

}

class JavaMapObjectSeq(javaMap: java.util.Map[_, _], locationString: () => String, converter: JavaValueConverter) extends LazyObjectSeq with SimpleObjectSeq with AlreadyMaterializedObjectSeq {

  private val cache = new mutable.HashMap[String, Value[_]]()

  override def removeKey(keyNameToRemove: QualifiedName)(implicit ctx: EvaluationContext): ObjectSeq = {
    val result = new util.LinkedHashMap[Any, Any](javaMap)
    result.remove(keyNameToRemove.name)
    new JavaMapObjectSeq(result, locationString, converter)
  }

  override def toIterator()(implicit ctx: EvaluationContext): Iterator[KeyValuePair] = {
    javaMap
      .entrySet()
      .iterator()
      .asScala
      .map((entry) => {
        KeyValuePair(KeyValue(entry.getKey.toString), entryToWeaveValue(entry))
      })
  }

  private def entryToWeaveValue(entry: java.util.Map.Entry[_, _])(implicit ctx: EvaluationContext): Value[_] = {
    val value = entry.getValue
    val name = entry.getKey.toString
    toWeaveValue(value, name)
  }

  def toWeaveValue(value: Any, name: String)(implicit ctx: EvaluationContext): Value[_] = {
    if (cache.isEmpty) {
      mapToWeaveValue(value, name)
    } else {
      val option = cache.get(name)
      if (option.isEmpty) {
        mapToWeaveValue(value, name)
      } else {
        option.get
      }
    }
  }

  private def mapToWeaveValue(value: Any, name: String)(implicit ctx: EvaluationContext): Value[_] = {
    val javaValue = toJavaValue(value, name)
    if (JavaValueHelper.isHeavyValue(javaValue)) {
      cache.put(name, javaValue)
    }
    javaValue
  }

  protected def toJavaValue(value: Any, name: String)(implicit ctx: EvaluationContext): Value[_] = {
    converter.convert(value, () => { locationString() + "." + name })
  }

  override def allKeyValuesOf(key: Value[QualifiedName])(implicit ctx: EvaluationContext): Option[ObjectSeq] = keyValueOf(key).map((kvp) => ObjectSeq(kvp))

  override def selectKeyValue(key: Value[QualifiedName])(implicit ctx: EvaluationContext): KeyValuePair = {
    val name = key.evaluate.name
    val value = javaMap.get(name)
    //Don't use optional map for performance reasons
    if (value == null) {
      if (javaMap.containsKey(name)) { //It may contain the key but with null value
        KeyValuePair(key, NullValue)
      } else {
        null
      }
    } else {
      KeyValuePair(key, toWeaveValue(value, name))
    }
  }

  override def size()(implicit ctx: EvaluationContext): Long = javaMap.size()

  override def isEmpty()(implicit ctx: EvaluationContext): Boolean = javaMap.isEmpty

}

object JavaMapObjectSeq {
  def apply(javaMap: java.util.Map[Any, Any], locationString: () => String, converter: JavaValueConverter): JavaMapObjectSeq = new JavaMapObjectSeq(javaMap, locationString, converter)
}
