package org.mule.weave.lsp

import org.eclipse.lsp4j.CodeLensOptions
import org.eclipse.lsp4j.CompletionOptions
import org.eclipse.lsp4j.DidChangeConfigurationParams
import org.eclipse.lsp4j.DidChangeWorkspaceFoldersParams
import org.eclipse.lsp4j.ExecuteCommandOptions
import org.eclipse.lsp4j.InitializeParams
import org.eclipse.lsp4j.InitializeResult
import org.eclipse.lsp4j.InitializedParams
import org.eclipse.lsp4j.ServerCapabilities
import org.eclipse.lsp4j.ServerInfo
import org.eclipse.lsp4j.SignatureHelpOptions
import org.eclipse.lsp4j.TextDocumentSyncKind
import org.eclipse.lsp4j.WorkspaceFolder
import org.eclipse.lsp4j.WorkspaceFoldersOptions
import org.eclipse.lsp4j.WorkspaceServerCapabilities
import org.eclipse.lsp4j.jsonrpc.services.JsonDelegate
import org.eclipse.lsp4j.services.LanguageClient
import org.eclipse.lsp4j.services.LanguageServer
import org.eclipse.lsp4j.services.WorkspaceService
import org.mule.dx.platform.api.protocol.ClientAware
import org.mule.weave.lsp.extension.client.LanguageClientDelegate
import org.mule.weave.lsp.extension.protocol.DataWeaveProtocolService
import org.mule.weave.lsp.extension.services.WeaveTextDocumentService
import org.mule.weave.lsp.jobs.JobManagerService
import org.mule.weave.lsp.jobs.Status
import org.mule.weave.lsp.project.DefaultProjectMetadata
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.ProjectKindContext
import org.mule.weave.lsp.project.ProjectKindFactory
import org.mule.weave.lsp.project.ProjectSettings
import org.mule.weave.lsp.services.DataWeaveDocumentService
import org.mule.weave.lsp.services.DiagnosticsPublisher
import org.mule.weave.lsp.services._
import org.mule.weave.lsp.services.delegate.TextDocumentServiceDelegate
import org.mule.weave.lsp.utils.URLUtils
import org.slf4j.LoggerFactory

import java.util
import java.util.concurrent.CompletableFuture
import java.util.concurrent.ExecutorService
import scala.collection.JavaConverters.seqAsJavaListConverter
import scala.collection.concurrent.TrieMap

class WeaveLanguageServer(projectKindFactory: ProjectKindFactory,
                          protocolService: DataWeaveProtocolService,
                          workspaceServiceManager: WorkspaceServiceManager,
                          jobManagerService: JobManagerService) extends LanguageServer with ClientAware[LanguageClient]
                          with WorkspaceServiceListener {

  private val logger = LoggerFactory.getLogger(classOf[WeaveLanguageServer])

  private val textDocumentService: TextDocumentServiceDelegate = new TextDocumentServiceDelegate()
  private val textDocumentServiceImpl: DataWeaveDocumentServiceDispatcher = new DataWeaveDocumentServiceDispatcher(fallbackDocumentService = (_) => languageServerDocumentService)

  private lazy val languageServerDocumentService: DataWeaveDocumentService = createLanguageServerDocumentService()

  private val projectKindsByWorkspaceFolderUri: TrieMap[String, ProjectKind] = TrieMap()

  private val languageClient: LanguageClientDelegate = new LanguageClientDelegate()

  private var clientLoggerFactory: ClientLoggerFactory = _
  private var clientLogger: ClientLogger = _

  private val settings: ProjectSettings = ProjectSettings()

  private val ID = "DATA_WEAVE_LSP"

  override def initialize(params: InitializeParams): CompletableFuture[InitializeResult] = {
    logger.debug(s"Initialize ${ID}")
    CompletableFuture.supplyAsync(() => {
      logger.debug(s"Initializing ${ID}")
      Runtime.getRuntime.addShutdownHook(new Thread(() => this.shutdown().get()))
      clientLoggerFactory = new ClientLoggerFactory(languageClient)
      clientLogger = clientLoggerFactory.createLogger(classOf[WeaveLanguageServer], ID)

      workspaceServiceManager.addWorkspaceServiceListeners(this)

      settings.load(params.getInitializationOptions)

      if (params.getWorkspaceFolders != null) {
        params.getWorkspaceFolders.forEach(workspaceFolder => discoverAndRegisterProjectKind(workspaceFolder))
      }
      textDocumentService.delegate = textDocumentServiceImpl

      workspaceServiceManager.diagnosticsPublisherService().setDefaultPublisher(new DiagnosticsPublisher(languageClient, Option.empty))
      workspaceServiceManager.workspaceEditService().setDefaultApplier(new WorkspaceEditApplier(languageClient, Option.empty))

      val commandManagerService = workspaceServiceManager.commandManagerService()
      val capabilities = new ServerCapabilities
      capabilities.setTextDocumentSync(TextDocumentSyncKind.Full)
      capabilities.setCompletionProvider(new CompletionOptions(true, java.util.Arrays.asList(".")))
      capabilities.setHoverProvider(true)
      capabilities.setDocumentSymbolProvider(true)
      capabilities.setDefinitionProvider(true)
      capabilities.setDocumentFormattingProvider(true)
      capabilities.setFoldingRangeProvider(true)
      capabilities.setCodeActionProvider(true)
      capabilities.setWorkspaceSymbolProvider(true)
      capabilities.setCodeLensProvider(new CodeLensOptions(true))
      capabilities.setRenameProvider(true)
      capabilities.setDocumentRangeFormattingProvider(true)
      capabilities.setReferencesProvider(true)
      capabilities.setSignatureHelpProvider(new SignatureHelpOptions(util.Arrays.asList("("), util.Arrays.asList(",")))
      capabilities.setExecuteCommandProvider(new ExecuteCommandOptions(commandManagerService.commandIds().asJava))

      val workspaceFoldersOptions = new WorkspaceFoldersOptions()
      workspaceFoldersOptions.setSupported(true)
      workspaceFoldersOptions.setChangeNotifications(true)
      val workspaceServerCapabilities = new WorkspaceServerCapabilities(workspaceFoldersOptions)
      capabilities.setWorkspace(workspaceServerCapabilities)

      val serverInfo = new ServerInfo("DataWeave", settings.languageLevelVersion.value())
      logger.debug(s"Initialize: Finished. Server: $serverInfo, Capabilities: $capabilities")
      new InitializeResult(capabilities, serverInfo)
    })
  }

  private def createProjectKind(projectMetadata: DefaultProjectMetadata, clientLoggerFactory: ClientLoggerFactory): ProjectKind = {
    val workspaceEditsService = workspaceServiceManager.workspaceEditService()

    val projectKind = projectKindFactory.createKind(
      ProjectKindContext(
        projectMetadata,
        clientLoggerFactory,
        protocolService.getClient(),
        jobManagerService,
        workspaceEditsService
      )
    )
    projectKind.initialize()
    projectMetadata.setEventBus(projectKind.eventBus())

    projectKind
  }

  override def initialized(params: InitializedParams): Unit = {
      projectKindsByWorkspaceFolderUri.foreach(entry => {
        jobManagerService
            .schedule((_: Status) => {
              initializedProjectKind(entry._2)
            }, s"Starting DataWeave Project [" + entry._2.project().getName + "]", "Starting DataWeave Project.")
        })
      clientLogger.logInfo("Initialized")
  }

  private def initializedProjectKind(projectKind: ProjectKind): Unit = {
    try {
      clientLogger.logInfo(s"Project: '${projectKind.project().getName()}' starting.")
      projectKind.initialized()
      clientLogger.logInfo(s"Project: '${projectKind.project().getName()}' started.")
    } catch {
      case e: Exception =>
        clientLogger.logError(s"Unable to start project: '${projectKind.project().getName()}'.", e)
    }
  }

  override def shutdown(): CompletableFuture[AnyRef] = {
    CompletableFuture.supplyAsync(() => {
      projectKindsByWorkspaceFolderUri.foreach(entry => {
        workspaceServiceManager.onProjectKindRemoved(entry._1)
        textDocumentServiceImpl.onProjectRemoved(entry._2)
        entry._2.shutdown()
      })
      logger.debug("Stopped")
      null
    })
  }

  override def exit(): Unit = {
    System.exit(0)
  }

  @JsonDelegate
  override def getTextDocumentService: WeaveTextDocumentService = {
    textDocumentService
  }

  override def getWorkspaceService: WorkspaceService = {
    workspaceServiceManager.workspaceService()
  }

  override def connect(client: LanguageClient): Unit = {
    languageClient.delegate = client
    workspaceServiceManager.onLanguageClientConnected(languageClient)
  }

  def executor(): ExecutorService = {
    IDEExecutors.defaultExecutor()
  }

  override def didChangeConfiguration(params: DidChangeConfigurationParams): Unit = {
    settings.update(params.getSettings)
  }

  override def didChangeWorkspaceFolders(params: DidChangeWorkspaceFoldersParams): Unit = {
    params.getEvent.getRemoved.forEach(removedWorkspaceFolder => {
      workspaceServiceManager.onProjectKindRemoved(removedWorkspaceFolder.getUri) match {
        case Some(projectKind) =>
          textDocumentServiceImpl.onProjectRemoved(projectKind)
          projectKind.shutdown()
        case None => logger.debug(s"There is no projectKind registered for URI: ${removedWorkspaceFolder.getUri}")
      }
    })

    params.getEvent.getAdded.forEach(addedWorkspaceFolder => {
      try {
        val projectKind: ProjectKind = discoverAndRegisterProjectKind(addedWorkspaceFolder)
        // Set initialized projectKind...
        initializedProjectKind(projectKind)
      } catch {
        case e: Exception =>
          clientLogger.logError(s"Unable to discover project for workspaceFolder: ${addedWorkspaceFolder.getUri}", e)
      }
    })
  }

  private def createLanguageServerDocumentService() = {
    val projectMetadata = DefaultProjectMetadata(Option.empty, settings)
    val projectKind = createProjectKind(projectMetadata, clientLoggerFactory)

    val languageServerDataWeaveDocumentService =
      new DataWeaveDocumentService(
        projectKind,
        executor())

    val uri = URLUtils.toLSPUrl(projectMetadata.home())
    workspaceServiceManager.standaloneContribution(uri, projectKind,
      new DefaultBaseWorkspaceServiceContributor(projectKind, projectMetadata, protocolService.getClient(), workspaceServiceManager.commandManagerService()))

    projectKindsByWorkspaceFolderUri.putIfAbsent(uri, projectKind)

    projectKind.initialize()
    projectMetadata.setEventBus(projectKind.eventBus())

    val currentContextClassLoader = Thread.currentThread.getContextClassLoader
    try {
      Thread.currentThread.setContextClassLoader(classOf[WeaveLanguageServer].getClassLoader)
      initializedProjectKind(projectKind)
    } finally {
      Thread.currentThread.setContextClassLoader(currentContextClassLoader)
    }

    languageServerDataWeaveDocumentService
  }

  private def discoverAndRegisterProjectKind(addedWorkspaceFolder: WorkspaceFolder) = {
    val projectMetadata = DefaultProjectMetadata(Option.apply(addedWorkspaceFolder.getUri), settings)
    val projectKind = createProjectKind(projectMetadata, clientLoggerFactory)

    textDocumentServiceImpl.onProjectKindCreated(projectKind,
      new DataWeaveDocumentService(
        projectKind,
        executor())
    )

    workspaceServiceManager.onProjectKindCreated(addedWorkspaceFolder.getUri, projectKind,
      new DefaultBaseWorkspaceServiceContributor(projectKind, projectMetadata, protocolService.getClient(), workspaceServiceManager.commandManagerService()))

    projectKindsByWorkspaceFolderUri.putIfAbsent(addedWorkspaceFolder.getUri, projectKind)
    projectKind
  }

}