package org.mule.weave.lsp.services

import org.mule.weave.extension.api.component.dependency.DependencyArtifact
import org.mule.weave.extension.api.component.dependency.WeaveDependencyComponent
import org.mule.weave.extension.api.project.ProjectMetadata
import org.mule.weave.lsp.extension.protocol.DependenciesParams
import org.mule.weave.lsp.extension.protocol.DependencyDefinition
import org.mule.weave.lsp.extension.protocol.DataWeaveProtocolClient
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.events.DependencyArtifactRemovedEvent
import org.mule.weave.lsp.project.events.DependencyArtifactResolvedEvent
import org.mule.weave.lsp.project.events.OnDependencyArtifactRemoved
import org.mule.weave.lsp.project.events.OnDependencyArtifactResolved
import org.mule.weave.lsp.utils.InternalEventBus
import org.mule.weave.lsp.utils.URLUtils

import java.util

class DataWeaveDependencyManagerService(weaveLanguageClient: DataWeaveProtocolClient) extends ToolingService  {

  var projectKind: ProjectKind = _

  override def initialize(projectKind: ProjectKind, eventBus: InternalEventBus): Unit = {
    this.projectKind = projectKind
    val dependencyManager: WeaveDependencyComponent = projectKind.dependencyManager()
    eventBus.register(DependencyArtifactResolvedEvent.ARTIFACT_RESOLVED, new OnDependencyArtifactResolved {
      override def onArtifactsResolved(artifacts: Array[DependencyArtifact]): Unit = {
        publishDependencies(projectKind, dependencyManager)
      }
    })

    eventBus.register(DependencyArtifactRemovedEvent.ARTIFACT_REMOVED, new OnDependencyArtifactRemoved {
      override def onArtifactsRemoved(artifacts: Array[DependencyArtifact]): Unit = {
        publishDependencies(projectKind, dependencyManager)
      }
    })
  }

  private def publishDependencies(projectKind: ProjectKind, dependencyManager: WeaveDependencyComponent): Unit = {
    val resolvedArtifacts: Array[DependencyArtifact] = dependencyManager.dependencies()
    val deps: Array[DependencyDefinition] = resolvedArtifacts.map((artifact) => {
      DependencyDefinition(id = artifact.groupId + ":" + artifact.artifactId + ":" + artifact.version, uri = artifact.file.toString)
    }).sortBy(dep => dep.id)
    weaveLanguageClient.publishDependencies(DependenciesParams(URLUtils.toLSPUrl(projectKind.projectMetadata.home()), util.Arrays.asList(deps: _*)))
  }

  override def initialized(): Unit = {
    publishDependencies(projectKind, projectKind.dependencyManager())
  }

}
