package org.mule.weave.lsp.utils

import org.mule.weave.extension.api.component.structure.WeaveModuleStructure
import org.mule.weave.extension.api.component.structure.WeaveProjectStructure
import org.mule.weave.extension.api.component.structure.WeaveRootKind
import org.mule.weave.lsp.project.components.ProjectStructureHelper
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.slf4j.LoggerFactory

import java.io.File

/**
  * Helper class that works with DataWeave well known directories and constants
  */
object WeaveDirectoryUtils {

  var verbose: Boolean = false
  val DWTest_FOLDER = "dw"
  val SAMPLES_FOLDER = "samples"
  val INPUTS_FOLDER = "inputs"
  val DW_TEST_RESOURCES = "resources"

  private val logger = LoggerFactory.getLogger(getClass)

  /**
    * Returns the DW home directory if exists it can be overwritten with env variable DW_HOME
    *
    * @return The home directory
    */
  def getDWHome(): File = {
    val homeUser = getUserHome
    val weavehome = System.getenv("DW_HOME")
    if (weavehome != null) {
      val home = new File(weavehome)
      home
    } else {
      if (verbose) {
        logger.info("'DW_HOME' env not set, trying home directory")
      }
      val defaultDWHomeDir = new File(homeUser, ".dw")
      if (!defaultDWHomeDir.exists()) {
        defaultDWHomeDir.mkdirs()
      }
      defaultDWHomeDir

    }
  }

  def inputsFolder(scenario: File) =
    new File(scenario, INPUTS_FOLDER)

  def getUserHome(): File = {
    new File(System.getProperty("user.home"))
  }

  /**
    * Returns the DW home directory if exists it can be overwritten with env variable DW_HOME
    *
    * @return The home directory
    */
  def getWorkingHome(): File = {
    val weavehome = System.getenv("DW_WORKING_PATH")
    if (weavehome != null) {
      val home = new File(weavehome)
      home
    } else {
      new File(getDWHome(), "tmp")
    }
  }

  /**
    * Returns the DW home directory if exists it can be overwritten with env variable DW_HOME
    *
    * @return The home directory
    */
  def getCacheHome(): File = {
    val weavehome = System.getenv("DW_CACHE_PATH")
    if (weavehome != null) {
      val home = new File(weavehome)
      home
    } else {
      new File(getDWHome(), "cache")
    }
  }

  /**
    * Returns the directory where all default jars are going to be present. It can be overwriten with DW_LIB_PATH
    *
    * @return The file
    */
  def getLibPathHome(): File = {
    val weavehome = System.getenv("DW_LIB_PATH")
    if (weavehome != null) {
      val home = new File(weavehome)
      home
    } else {
      new File(getDWHome(), "libs")
    }
  }

  def sanitizeFilename(inputName: String): String = {
    inputName.replaceAll("[^a-zA-Z0-9-_\\.]", "_")
  }

  def toFolderName(nameIdentifier: NameIdentifier): String = {
    val parts = nameIdentifier.name.split(NameIdentifier.SEPARATOR)
    val sanitizedParts = parts.map(sanitizeFilename)
    val folder = sanitizedParts.mkString("/")
    folder
  }

  def wtfUnitTestFolder(projectStructure: WeaveProjectStructure): Array[File] = {
    projectStructure.modules
      .flatMap((m) => {
        m.roots.flatMap((root) => {
          if (root.kind == WeaveRootKind.TEST) {
            root.sources.find((f) => {
              f.getName.equals(WeaveDirectoryUtils.DWTest_FOLDER)
            })
          } else {
            None
          }
        })
      })
  }

  def wtfResourcesTestFolder(projectStructure: WeaveProjectStructure): Array[File] = {
    projectStructure.modules.flatMap(moduleResourcesTestFolder)
  }

  def wtfResourcesTestFolderForFile(projectStructure: WeaveProjectStructure, uri: String): Array[File] = {
    val maybeModule = ProjectStructureHelper.moduleOfFile(projectStructure, uri)
    maybeModule.map(moduleResourcesTestFolder).getOrElse(Array.empty)
  }

  private def moduleResourcesTestFolder(module: WeaveModuleStructure): Array[File] = module.roots.flatMap(root => {
    if (root.kind == WeaveRootKind.TEST) {
      root.resources.find((f) => {
        f.getName.equals(WeaveDirectoryUtils.DW_TEST_RESOURCES)
      })
    } else {
      None
    }
  })


  def wtfUnitDefaultTestSourceFolder(projectStructure: WeaveProjectStructure): Array[File] = {
    projectStructure.modules
      .flatMap((m) => {
        m.roots.flatMap((root) => {
          if (root.kind == WeaveRootKind.TEST) {
            Some(root.defaultSourceFolder)
          } else {
            None
          }
        })
      })
  }
}

