package org.mule.weave.lsp.vfs

import org.mule.weave.extension.api.component.dependency.DependencyArtifact
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.events.DependencyArtifactRemovedEvent
import org.mule.weave.lsp.project.events.DependencyArtifactResolvedEvent
import org.mule.weave.lsp.project.events.OnDependencyArtifactRemoved
import org.mule.weave.lsp.project.events.OnDependencyArtifactResolved
import org.mule.weave.lsp.services.ToolingService
import org.mule.weave.lsp.utils.InternalEventBus
import org.mule.weave.lsp.vfs.events.LibrariesAddedEvent
import org.mule.weave.lsp.vfs.events.LibrariesRemovedEvent
import org.mule.weave.v2.editor.ChangeListener
import org.mule.weave.v2.editor.VirtualFile
import org.mule.weave.v2.editor.VirtualFileSystem
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.sdk.WeaveResourceResolver
import org.slf4j.LoggerFactory

import java.util
import scala.collection.JavaConverters.asJavaIteratorConverter
import scala.collection.JavaConverters.asScalaIteratorConverter
import scala.collection.mutable

/**
  * A virtual file system that handles Maven Libraries. This VFS allows to load and unload libraries.
  *
  * @param maven                The Maven Dependency Manager
  * @param messageLoggerService The user logger
  */
class LibrariesVirtualFileSystem(jarFileNameIdentifierResolver: JarFileNameIdentifierResolver) extends VirtualFileSystem with ToolingService {

  private val logger = LoggerFactory.getLogger(getClass)

  private val libraries: mutable.Map[String, ArtifactVirtualFileSystem] = new mutable.HashMap()
  private var eventBus: InternalEventBus = _

  override def initialize(projectKind: ProjectKind, eventBus: InternalEventBus): Unit = {
    this.eventBus = eventBus
    eventBus.register(DependencyArtifactResolvedEvent.ARTIFACT_RESOLVED, new OnDependencyArtifactResolved {
      override def onArtifactsResolved(artifacts: Array[DependencyArtifact]): Unit = {
        val resolvedArtifacts = artifacts.map((artifact) => {
          val libraryVFS: ArtifactVirtualFileSystem = if (!artifact.file.isDirectory) {
            new JarVirtualFileSystem(artifact.artifactId, artifact.file, jarFileNameIdentifierResolver)
          } else {
            new FolderVirtualFileSystem(artifact.artifactId, artifact.file)
          }
          (artifact.artifactId, libraryVFS)
        })
        addLibrary(resolvedArtifacts)
      }
    })

    eventBus.register(DependencyArtifactRemovedEvent.ARTIFACT_REMOVED, new OnDependencyArtifactRemoved {
      override def onArtifactsRemoved(artifacts: Array[DependencyArtifact]): Unit = {
        removeLibraries(artifacts)
      }
    })
  }

  override def file(path: String): VirtualFile = {
    logger.debug(s"file ${path}")
    libraries
      .toStream
      .flatMap((vfs) => {
        logger.debug("Module:" + vfs._1)
        Option(vfs._2.file(path))
      })
      .headOption
      .orNull
  }


  private def removeLibraries(artifacts: Array[DependencyArtifact]): Unit = {
    val removedArtifacts = artifacts.map(artifact => {
      logger.debug(s"Artifact `${artifact.artifactId}` was removed.")
      libraries.remove(artifact.artifactId)
    }).filter(maybeArtifacts => maybeArtifacts.isDefined).map(maybeArtifact => maybeArtifact.get)
    eventBus.fire(new LibrariesRemovedEvent(removedArtifacts))
  }

  def addLibrary(libs: Array[(String, ArtifactVirtualFileSystem)]): Unit = {
    libs.foreach((lib) => {
      val virtualFileSystem: ArtifactVirtualFileSystem = lib._2
      val name: String = lib._1
      this.libraries.put(name, virtualFileSystem)
      logger.debug(s"Artifact `${virtualFileSystem.artifactId()}` was resolved.")
    })
    eventBus.fire(new LibrariesAddedEvent(this.libraries.values.toArray))
  }

  def getLibrary(name: String): VirtualFileSystem = {
    libraries.get(name).orNull
  }

  def getLibraries(): mutable.Map[String, ArtifactVirtualFileSystem] = {
    libraries
  }

  override def changeListener(cl: ChangeListener): Unit = {
    libraries.foreach(_._2.changeListener(cl))
  }

  override def onChanged(virtualFile: VirtualFile): Unit = {
    libraries.foreach(_._2.onChanged(virtualFile))
  }

  override def removeChangeListener(service: ChangeListener): Unit = {
    libraries.foreach(_._2.removeChangeListener(service))
  }

  override def asResourceResolver: WeaveResourceResolver = {
    new LibrariesWeaveResourceResolver()
  }


  override def listFiles(): util.Iterator[VirtualFile] = {
    libraries.values.toIterator.flatMap(_.listFiles().asScala).asJava
  }


  class LibrariesWeaveResourceResolver() extends WeaveResourceResolver {

    override def resolve(name: NameIdentifier): Option[WeaveResource] = {
      val resolvers: Iterator[WeaveResourceResolver] = resourceResolver
      while (resolvers.hasNext) {
        val resolver = resolvers.next()
        val resolved: Option[WeaveResource] = resolver.resolve(name)
        if (resolved.isDefined) {
          return resolved
        }
      }
      None
    }

    override def resolvePath(path: String): Option[WeaveResource] = {
      val resolvers: Iterator[WeaveResourceResolver] = resourceResolver
      while (resolvers.hasNext) {
        val resolver = resolvers.next()
        val resolved: Option[WeaveResource] = resolver.resolvePath(path)
        if (resolved.isDefined) {
          return resolved
        }
      }
      None
    }

    override def resolveAll(name: NameIdentifier): Seq[WeaveResource] = {
      resourceResolver.flatMap(_.resolveAll(name)).toSeq
    }
  }

  private def resourceResolver: Iterator[WeaveResourceResolver] = {
    libraries.values.iterator.map(_.asResourceResolver)
  }
}


