package org.mule.weave.lsp.commands

import org.eclipse.lsp4j.ApplyWorkspaceEditParams
import org.eclipse.lsp4j.Command
import org.eclipse.lsp4j.ExecuteCommandParams
import org.eclipse.lsp4j.MessageParams
import org.eclipse.lsp4j.MessageType
import org.eclipse.lsp4j.ResourceOperation
import org.eclipse.lsp4j.TextDocumentEdit
import org.eclipse.lsp4j.VersionedTextDocumentIdentifier
import org.eclipse.lsp4j.WorkspaceEdit
import org.eclipse.lsp4j.jsonrpc.messages.Either
import org.mule.weave.extension.api.extension.command.WeaveCommand
import org.mule.weave.lsp.services.DataWeaveToolingService
import org.mule.weave.lsp.services.UIService
import org.mule.weave.lsp.services.WorkspaceServiceManager

import java.util

class ExtractVariableCommand(workspaceServiceManager: WorkspaceServiceManager, uIService: UIService) extends WeaveCommand {
  override def commandId(): String = Commands.DW_EXTRACT_VARIABLE

  override def execute(params: ExecuteCommandParams): AnyRef = {
    val startOffset = Commands.argAsInt(params.getArguments, 0)
    val endOffset = Commands.argAsInt(params.getArguments, 1)
    val uri = Commands.argAsString(params.getArguments, 2)

    val maybeProjectKind = workspaceServiceManager.projectKind(uri)
    if (maybeProjectKind.isDefined) {
      val projectKind = maybeProjectKind.get
      val toolingService = projectKind.toolingService(classOf[DataWeaveToolingService]).openDocument(uri)
      val maybeRefactor = toolingService.extractVariable(startOffset, endOffset)
      maybeRefactor match {
        case Some(refactor) => {
          val document = new LSPWeaveTextDocument(toolingService)
          refactor.run(document, refactor.parameters().map((p) => (p.name, p.defaultValue)).toMap)
          val edit = new WorkspaceEdit()
          val documentEdit = new TextDocumentEdit(new VersionedTextDocumentIdentifier(uri, 0), document.edits())
          val documentChanges: util.List[Either[TextDocumentEdit, ResourceOperation]] = new util.ArrayList[Either[TextDocumentEdit, ResourceOperation]]()
          documentChanges.add(Either.forLeft(documentEdit))
          edit.setDocumentChanges(documentChanges)
          val response = workspaceServiceManager.workspaceEditService.applyEdit(new ApplyWorkspaceEditParams(edit, "Extract Variable")).get()
          response.isApplied
        }
        case None =>
      }
    } else {
      uIService.showMessage(new MessageParams(MessageType.Error, s"No project is associated to: ${uri}. ${commandId} cannot be executed"))
    }
    null
  }


  override def name(): String = "Extract variable."

  override def description(params: ExecuteCommandParams): String = "Extracting variable."
}

object ExtractVariableCommand {
  def createCommand(startOffset: Int, endOffset: Int, uri: String): Command = {
    new Command("Extract Variable",Commands.DW_EXTRACT_VARIABLE,
      util.Arrays.asList[AnyRef](
        startOffset: java.lang.Integer,
        endOffset: java.lang.Integer,
        uri
      ))
  }
}
