package org.mule.weave.lsp.services

import org.eclipse.lsp4j.Diagnostic
import org.eclipse.lsp4j.DiagnosticSeverity
import org.eclipse.lsp4j.PublishDiagnosticsParams
import org.mule.weave.extension.api.extension.validation.DocumentNode
import org.mule.weave.extension.api.extension.validation.MessageBuilderFactory
import org.mule.weave.extension.api.extension.validation.ValidationTypeLevel
import org.mule.weave.extension.api.extension.validation.WeaveValidator
import org.mule.weave.extension.api.metadata.ContextMetadata
import org.mule.weave.extension.api.metadata.InputMetadata
import org.mule.weave.extension.api.project.ProjectMetadata
import org.mule.weave.lsp.indexer.events.IndexingFinishedEvent
import org.mule.weave.lsp.indexer.events.IndexingType
import org.mule.weave.lsp.indexer.events.IndexingType.IndexingType
import org.mule.weave.lsp.indexer.events.OnIndexingFinished
import org.mule.weave.lsp.project.ProjectKind
import org.mule.weave.lsp.project.Settings
import org.mule.weave.lsp.project.events.OnProjectStarted
import org.mule.weave.lsp.project.events.OnSettingsChanged
import org.mule.weave.lsp.project.events.ProjectStartedEvent
import org.mule.weave.lsp.project.events.SettingsChangedEvent
import org.mule.weave.lsp.utils.InternalEventBus
import org.mule.weave.lsp.utils.LSPConverters.toDiagnostic
import org.mule.weave.lsp.utils.LSPConverters.toDiagnosticKind
import org.mule.weave.lsp.utils.URLUtils
import org.mule.weave.lsp.utils.VFUtils
import org.mule.weave.lsp.utils.WeaveTypeUtils
import org.mule.weave.lsp.vfs.VirtualFileAdapter
import org.mule.weave.lsp.vfs.events.OnProjectVirtualFileChangedEvent
import org.mule.weave.lsp.vfs.events.OnProjectVirtualFileCreatedEvent
import org.mule.weave.lsp.vfs.events.OnProjectVirtualFileDeletedEvent
import org.mule.weave.lsp.vfs.events.ProjectVirtualFileChangedEvent
import org.mule.weave.lsp.vfs.events.ProjectVirtualFileCreatedEvent
import org.mule.weave.lsp.vfs.events.ProjectVirtualFileDeletedEvent
import org.mule.weave.v2.api.tooling.impl.message.DefaultMessageBuilder
import org.mule.weave.v2.api.tooling.message.MessageBuilder
import org.mule.weave.v2.api.tooling.ts.DWType
import org.mule.weave.v2.editor.ImplicitInput
import org.mule.weave.v2.editor.MappingInput
import org.mule.weave.v2.editor.QuickFix
import org.mule.weave.v2.editor.ValidationMessage
import org.mule.weave.v2.editor.ValidationMessages
import org.mule.weave.v2.editor.VirtualFile
import org.mule.weave.v2.editor.VirtualFileSystem
import org.mule.weave.v2.editor.WeaveDocumentToolingService
import org.mule.weave.v2.editor.WeaveToolingService
import org.mule.weave.v2.parser.MessageCategory
import org.mule.weave.v2.parser.ScopePhaseCategory
import org.mule.weave.v2.parser.TypePhaseCategory
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.versioncheck.SVersion
import org.slf4j.LoggerFactory

import java.util
import java.util.Optional
import java.util.concurrent.CompletableFuture
import java.util.concurrent.Executor
import javax.annotation.Nullable

class DataWeaveToolingService(val project: ProjectMetadata,
                              val vfs: VirtualFileSystem,
                              publisherService: DiagnosticsPublisherService,
                              documentServiceFactory: WeaveToolingServiceFactory,
                              executor: Executor) extends ToolingService {

  private val logger = LoggerFactory.getLogger(getClass)
  protected var projectKind: ProjectKind = _
  private lazy val _documentService: WeaveToolingService = documentServiceFactory.create()
  private var customValidators: Array[WeaveValidator] = _

  @volatile
  protected var indexed: Boolean = false

  override def initialize(projectKind: ProjectKind, eventBus: InternalEventBus): Unit = {
    this.projectKind = projectKind
    this.customValidators = projectKind.customValidators()
    eventBus.register(SettingsChangedEvent.SETTINGS_CHANGED, new OnSettingsChanged {
      override def onSettingsChanged(modifiedSettingsName: Array[String]): Unit = {
        if (modifiedSettingsName.contains(Settings.LANGUAGE_LEVEL_PROP_NAME)) {
          validateAllEditors("settingsChanged")
        }
      }
    })

    eventBus.register(IndexingFinishedEvent.INDEXING_FINISHED, new OnIndexingFinished() {
      override def onIndexingFinished(indexingType: IndexingType): Unit = {
        if (IndexingType.Dependencies.equals(indexingType)) {
          indexed = true
          validateAllEditors("indexingFinishes")
        }
      }
    })

    eventBus.register(ProjectStartedEvent.PROJECT_STARTED, new OnProjectStarted {
      override def onProjectStarted(projectMetadata: ProjectMetadata): Unit = {
        validateAllEditors("projectStarted")
      }
    })

    eventBus.register(ProjectVirtualFileCreatedEvent.VIRTUAL_FILE_CREATED, new OnProjectVirtualFileCreatedEvent {
      override def onVirtualFileCreated(vf: VirtualFile): Unit = {
        if (VFUtils.isDWFile(vf.url())) {
          validateFile(vf, "onCreated")
        } else {
          // Validate file if a scenario is modified
          findCorrespondingDwFile(vf.url()).foreach(dwVirtualFile => validateFile(dwVirtualFile, "onScenarioChanged"))
        }
      }
    })

    eventBus.register(ProjectVirtualFileChangedEvent.VIRTUAL_FILE_CHANGED, new OnProjectVirtualFileChangedEvent {
      override def onVirtualFileChanged(vf: VirtualFile): Unit = {
        if (VFUtils.isDWFile(vf.url())) {
          validateFile(vf, "onChanged")
        } else {
          // Validate file if a scenario is modified
          findCorrespondingDwFile(vf.url()).foreach(dwVirtualFile => validateFile(dwVirtualFile, "onScenarioChanged"))
        }
      }
    })

    eventBus.register(ProjectVirtualFileDeletedEvent.VIRTUAL_FILE_DELETED, new OnProjectVirtualFileDeletedEvent {
      override def onVirtualFileDeleted(vf: VirtualFile): Unit = {
        if (VFUtils.isDWFile(vf.url())) {
          validateDependencies(vf, "onDeleted")
          publisherService.publishDiagnostics(new PublishDiagnosticsParams(vf.url(), new util.ArrayList[Diagnostic]()))
        } else {
          // Validate file if a scenario is modified
          findCorrespondingDwFile(vf.url()).foreach(dwVirtualFile => validateFile(dwVirtualFile, "onScenarioChanged"))
        }
      }
    })
  }

  private def findCorrespondingDwFile(str: String): Option[VirtualFile] = {
    val sampleDataManager = projectKind.sampleDataManager()
    documentService().openEditors().find((oe) => {
      URLUtils.isChildOfAny(str, sampleDataManager.listScenarios(oe.file.getNameIdentifier).map(scenario => scenario.file))
    }).map(weaveDocToolingService => weaveDocToolingService.file)
  }

  private def validateAllEditors(reason: String): Unit = {
    val services = documentService().openEditors().map(openEditor => openEditor.file.url());
    documentService().invalidateAll()
    services.foreach((url) => {
      triggerValidation(url, reason)
    })
  }


  private def validateDependencies(vf: VirtualFile, reason: String): Unit = {
    val fileLogicalName: NameIdentifier = vf.getNameIdentifier
    val dependants: Seq[NameIdentifier] = documentService().dependantsOf(fileLogicalName)
    logger.debug(s"Validate dependants of $fileLogicalName: ${dependants.mkString("[", ",", "]")}")
    dependants.foreach((ni) => {
      vf.fs().asResourceResolver.resolve(ni) match {
        case Some(resource) => {
          triggerValidation(resource.url(), "dependantChanged ->" + reason + s" ${vf.url()}")
        }
        case None => {
          logger.warn("No resource found for file " + vf.url())
        }
      }
    })
  }

  def loadType(typeString: String): Option[WeaveType] = {
    documentService().loadType(typeString)
  }

  private def documentService(): WeaveToolingService = {
    _documentService
  }

  /**
   * Returns the input directive that has the same name as the one specified in the specified document.
   *
   * @param uri       The uri of the document to search in
   * @param inputName The name of the input
   * @return The input directive if any
   */
  def inputOf(uri: String, inputName: String): Option[MappingInput] = {
    val documentToolingService = openDocument(uri)
    documentToolingService.inputOf(inputName)
  }

  def contextMetadataFor(uri: String): Optional[ContextMetadata] = {
    val vf = vfs.file(uri)
    if (vf != null) {
      projectKind.metadataProvider().metadata(VirtualFileAdapter(vf))
    } else {
      Optional.empty()
    }
  }

  def openDocument(uri: String): WeaveDocumentToolingService = {
    val maybeContext = contextMetadataFor(uri)
    if (maybeContext.isPresent) {
      val context = maybeContext.get()
      openDocument(uri, context.getInputMetadata(), context.getOutput())
    } else {
      openDocumentWithoutContext(uri)
    }
  }

  def openDocument(uri: String, maybeContext: Optional[ContextMetadata]): WeaveDocumentToolingService = {
    if (maybeContext.isPresent) {
      val context = maybeContext.get()
      openDocument(uri, context.getInputMetadata, context.getOutput)
    } else {
      openDocument(uri)
    }
  }

  def openDocumentWithoutContext(uri: String): WeaveDocumentToolingService = {
    _documentService.open(uri, ImplicitInput(), Option.empty)
  }

  private def openDocument(uri: String, @Nullable input: InputMetadata, @Nullable expectedOutput: DWType): WeaveDocumentToolingService = {
    val output = if (expectedOutput != null) {
      if (!expectedOutput.isInstanceOf[WeaveType]) {
        throw new IllegalArgumentException(s"Unexpected `expectedOutput` instance, please, contact the extension owner. Actual: ${expectedOutput.getClass.getName}")
      }
      Some(expectedOutput.asInstanceOf[WeaveType])
    } else {
      None
    }
    _documentService.open(uri, WeaveTypeUtils.toImplicitInput(input), output)
  }

  def closeDocument(uri: String): Unit = {
    documentService().close(uri)
  }

  def withLanguageLevel(dwLanguageLevel: String): WeaveToolingService = {
    _documentService.updateLanguageLevel(SVersion.fromString(dwLanguageLevel))
  }


  def validateFile(vf: VirtualFile, reason: String): Unit = {
    triggerValidation(vf.url(), reason, () => validateDependencies(vf, reason))
  }

  /**
   * Triggers the validation of the specified document.
   *
   * @param documentUri        The URI to be validated
   * @param onValidationFinish A Callback that is called when the validation finishes
   */
  def triggerValidation(documentUri: String, reason: String, onValidationFinish: () => Unit = () => {}): Unit = {
    logger.debug("TriggerValidation of: " + documentUri + " reason " + reason)
    CompletableFuture.runAsync(() => {
      logger.debug(s"[${Thread.currentThread().getName}][${this.getClass.getName}] Init TriggerValidation of: " + documentUri + " reason " + reason)
      withLanguageLevel(projectKind.dependencyManager().dwVersion())
      val messages: ValidationMessages = validate(documentUri)
      val diagnostics = toDiagnostics(messages)
      logger.debug(s"[${Thread.currentThread().getName}][${this.getClass.getName}] TriggerValidation finished: " + documentUri + " reason " + reason + s". Diagnostics: [${diagnostics}]")
      publisherService.publishDiagnostics(new PublishDiagnosticsParams(documentUri, diagnostics))
      onValidationFinish()

    }, executor)
  }

  def toDiagnostics(messages: ValidationMessages): util.List[Diagnostic] = {
    val diagnostics = new util.ArrayList[Diagnostic]
    messages.errorMessage.foreach(message => {
      diagnostics.add(toDiagnostic(message, DiagnosticSeverity.Error))
    })

    messages.warningMessage.foreach((message) => {
      diagnostics.add(toDiagnostic(message, DiagnosticSeverity.Warning))
    })
    diagnostics
  }

  def quickFixesFor(documentUri: String, startOffset: Int, endOffset: Int, kind: String, severity: String, maybeContext: Optional[ContextMetadata] = Optional.empty()): Array[QuickFix] = {
    val messages: ValidationMessages = validate(documentUri, maybeContext)
    val messageFound: Option[ValidationMessage] = if (severity == DiagnosticSeverity.Error.name()) {
      messages.errorMessage.find((m) => {
        matchesMessage(m, kind, startOffset, endOffset)
      })
    } else {
      messages.warningMessage.find((m) => {
        matchesMessage(m, kind, startOffset, endOffset)
      })
    }
    messageFound.map(_.quickFix).getOrElse(Array.empty)
  }

  private def matchesMessage(m: ValidationMessage, kind: String, startOffset: Int, endOffset: Int): Boolean = {
    m.location.startPosition.index == startOffset && m.location.endPosition.index == endOffset && toDiagnosticKind(m) == kind
  }

  /**
   * Executes Weave Validation into the corresponding type level
   *
   * @param documentUri The URI to be validates
   * @return The Validation Messages
   */
  def validate(documentUri: String, maybeContext: Optional[ContextMetadata] = Optional.empty()): ValidationMessages = {
    val documentToolingService = if (maybeContext.isPresent) {
      val context = maybeContext.get
      openDocument(documentUri, context.getInputMetadata, context.getOutput)
    } else {
      val maybeContext = contextMetadataFor(documentUri)
      if (maybeContext.isPresent) {
        val context = maybeContext.get
        openDocument(documentUri, context.getInputMetadata, null)
      } else {
        openDocumentWithoutContext(documentUri)
      }
    }
    val messages: ValidationMessages =
      if (indexed && Settings.isTypeLevel(project.settings)) {
        documentToolingService.typeCheck()
          .concat(applyCustomValidations(documentUri, documentToolingService, ValidationTypeLevel.SCOPE))
          .concat(applyCustomValidations(documentUri, documentToolingService, ValidationTypeLevel.TYPE))
      } else if (indexed && Settings.isScopeLevel(project.settings)) {
        documentToolingService.scopeCheck()
          .concat(applyCustomValidations(documentUri, documentToolingService, ValidationTypeLevel.SCOPE))
      } else {
        documentToolingService.parseCheck()
      }
    messages
  }

  private def applyCustomValidations(documentUri: String, documentToolingService: WeaveDocumentToolingService, typeLevel: ValidationTypeLevel): ValidationMessages = {
     documentToolingService.ast()
       .map(_.dwAstNode)
       .map(rootDwAstNode => {
         var category: MessageCategory = TypePhaseCategory
         if (typeLevel == ValidationTypeLevel.SCOPE) {
           category = ScopePhaseCategory
         }
         val messageCollector: MessageValidationCollectorAdapter = MessageValidationCollectorAdapter(category)
          customValidators
            .filter(v => v.appliesTo(documentUri))
            .filter(v => v.validationLevel() == typeLevel)
            .foreach(v => {
              try {
                v.validate(new DocumentNode(documentUri , rootDwAstNode), messageCollector, new MessageBuilderFactory() {
                  override def createMessageBuilder(): MessageBuilder = new DefaultMessageBuilder()
                })
              } catch {
                case e: Exception => getLogger().error(s"There was an error while executing custom validator: ${v.getClass}", e)
              }
            })
          messageCollector.getValidationMessages
        })
       .getOrElse(ValidationMessages(Array(), Array()))
  }

  def getLogger() = logger
}


trait WeaveToolingServiceFactory {
  def create(): WeaveToolingService
}